package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmBannerAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.UsersSegmentsValidationUtil;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.minusKeywordsNotAllowed;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class CpmBannerAdGroupValidation implements AdGroupTypeSpecificValidationService<CpmBannerAdGroup> {
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;
    private final AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory;

    @Autowired
    public CpmBannerAdGroupValidation(
            ShardHelper shardHelper, CampaignRepository campaignRepository,
            AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory) {
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.priceSalesValidatorFactory = priceSalesValidatorFactory;
    }

    @Override
    public ValidationResult<List<ModelChanges<CpmBannerAdGroup>>, Defect> validateModelChanges(ClientId clientId,
            List<ModelChanges<CpmBannerAdGroup>> modelChangesList) {
        int shard = shardHelper.getShardByClientId(clientId);
        var adGroupIds = listToSet(modelChangesList, ModelChanges::getId);
        var campaignWithTypeByAdGroupId =
                campaignRepository.getCampaignsWithTypeByAdGroupIds(shard, clientId, adGroupIds);

        AdGroupPriceSalesModelChangesValidator<CpmBannerAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createModelChangesValidator(shard, campaignWithTypeByAdGroupId);

        return ListValidationBuilder.<ModelChanges<CpmBannerAdGroup>, Defect>of(modelChangesList)
                .checkEachBy(priceSalesValidator, When.valueIs(
                        changes -> campaignWithTypeByAdGroupId.get(changes.getId()).getType() == CampaignType.CPM_PRICE
                ))
                .getResult();
    }

    @Override
    public ValidationResult<List<CpmBannerAdGroup>, Defect> validateAdGroups(ClientId clientId,
            List<CpmBannerAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        var priceSalesValidator = priceSalesValidatorFactory.createBeanValidator();

        return ListValidationBuilder.<CpmBannerAdGroup, Defect>of(adGroups)
                .checkEachBy(this::commonCpmBannerAdGroupValidation)
                .checkEachBy(adGroup -> validateCpmPriceBannerAdGroup(adGroup, priceSalesValidator),
                        When.valueIs(adGroup -> campaignTypes.get(adGroup.getCampaignId()) == CampaignType.CPM_PRICE))
                .getResult();
    }

    private ValidationResult<CpmBannerAdGroup, Defect> validateCpmPriceBannerAdGroup(
            CpmBannerAdGroup adGroup,
            AdGroupPriceSalesValidator priceSalesValidator) {

        ModelItemValidationBuilder<CpmBannerAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.checkBy(priceSalesValidator);

        return vb.getResult();
    }

    private ValidationResult<CpmBannerAdGroup, Defect> commonCpmBannerAdGroupValidation(CpmBannerAdGroup adGroup) {
        ModelItemValidationBuilder<CpmBannerAdGroup> vb = ModelItemValidationBuilder.of(adGroup);
        boolean isKeywordCriterionType = adGroup.getCriterionType() == CriterionType.KEYWORD;

        UsersSegmentsValidationUtil.validateAdGroupWithUsersSegments(vb, adGroup.getType(), false);

        vb.item(AdGroup.MINUS_KEYWORDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()),
                        When.isFalse(isKeywordCriterionType));
        vb.item(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()),
                        When.isFalse(isKeywordCriterionType));

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<CpmBannerAdGroup>, Defect> validateAddAdGroups(
            ClientId clientId,
            List<CpmBannerAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        AdGroupPriceSalesAddValidator<CpmBannerAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createAddValidator(shard, campaignTypes, adGroups);

        return ListValidationBuilder.<CpmBannerAdGroup, Defect>of(adGroups)
                .checkEachBy(priceSalesValidator, When.valueIs(
                        adGroup -> campaignTypes.get(adGroup.getCampaignId()) == CampaignType.CPM_PRICE
                ))
                .getResult();
    }

    @Override
    public Class<CpmBannerAdGroup> getAdGroupClass() {
        return CpmBannerAdGroup.class;
    }
}
