package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmOutdoorAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.UsersSegmentsValidationUtil;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.minusKeywordsNotAllowed;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.notAllowedValue;
import static ru.yandex.direct.regions.Region.RUSSIA_REGION_ID;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class CpmOutdoorAdGroupValidation extends BaseAdGroupTypeSpecificValidationService<CpmOutdoorAdGroup> {

    public static final int PAGE_BLOCKS_MAX = 2000;
    public static final List<Long> OUTDOOR_GEO_DEFAULT = singletonList(RUSSIA_REGION_ID);

    private final AdGroupPageBlocksValidatorFactory adGroupPageBlocksValidatorFactory;
    private final FeatureService featureService;

    @Autowired
    public CpmOutdoorAdGroupValidation(AdGroupPageBlocksValidatorFactory adGroupPageBlocksValidatorFactory,
                                       FeatureService featureService) {
        this.adGroupPageBlocksValidatorFactory = adGroupPageBlocksValidatorFactory;
        this.featureService = featureService;
    }

    @Override
    public ValidationResult<List<CpmOutdoorAdGroup>, Defect> validateAdGroups(ClientId clientId,
                                                                              List<CpmOutdoorAdGroup> adGroups) {
        // Используем тип без generic-аргументов, чтобы не использовать PropHolder-тип,
        // который запрещён к употреблению в коде. Но он выводится компилятором, и всё ок.
        AdGroupPageBlocksValidator pageBlocksValidator =
                adGroupPageBlocksValidatorFactory.createValidator(
                        adGroups, CpmOutdoorAdGroup.PAGE_BLOCKS, PlacementType.OUTDOOR, PAGE_BLOCKS_MAX,
                        featureService.isEnabledForClientId(clientId, FeatureName.OUTDOOR_INDOOR_TESTING_PAGES));

        return ListValidationBuilder.<CpmOutdoorAdGroup, Defect>of(adGroups)
                .checkEachBy(adGroup -> validateAdGroup(adGroup, pageBlocksValidator))
                .getResult();
    }

    public ValidationResult<CpmOutdoorAdGroup, Defect> validateAdGroup(
            CpmOutdoorAdGroup adGroup,
            AdGroupPageBlocksValidator pageBlocksValidator) {
        ModelItemValidationBuilder<CpmOutdoorAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        UsersSegmentsValidationUtil.validateAdGroupWithUsersSegments(vb, adGroup.getType(), false);

        vb.item(AdGroup.GEO)
                .check(valueIsAllowed(OUTDOOR_GEO_DEFAULT));

        vb.item(AdGroup.MINUS_KEYWORDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));
        vb.item(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));

        vb.checkBy(pageBlocksValidator);

        return vb.getResult();
    }

    @Override
    public Class<CpmOutdoorAdGroup> getAdGroupClass() {
        return CpmOutdoorAdGroup.class;
    }

    private <T> Constraint<T, Defect> valueIsAllowed(T allowedValue) {
        return fromPredicate(allowedValue::equals, notAllowedValue());
    }

    @Override
    ValidationResult<CpmOutdoorAdGroup, Defect> validateAdGroup(CpmOutdoorAdGroup adGroup) {
        throw new UnsupportedOperationException("this method must not be called");
    }
}
