package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmVideoAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.suboperation.UsersSegmentsValidationUtil;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.minusKeywordsNotAllowed;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class CpmVideoAdGroupValidation implements AdGroupTypeSpecificValidationService<CpmVideoAdGroup> {

    private final FeatureService featureService;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory;

    @Autowired
    public CpmVideoAdGroupValidation(FeatureService featureService,
                                     ShardHelper shardHelper,
                                     CampaignRepository campaignRepository,
                                     AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory) {
        this.featureService = featureService;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.priceSalesValidatorFactory = priceSalesValidatorFactory;
    }

    @Override
    public ValidationResult<List<ModelChanges<CpmVideoAdGroup>>, Defect> validateModelChanges(ClientId clientId,
                                                                                              List<ModelChanges<CpmVideoAdGroup>> modelChangesList) {
        int shard = shardHelper.getShardByClientId(clientId);
        var adGroupIds = listToSet(modelChangesList, ModelChanges::getId);
        var campaignWithTypeByAdGroupId =
                campaignRepository.getCampaignsWithTypeByAdGroupIds(shard, clientId, adGroupIds);

        AdGroupPriceSalesModelChangesValidator<CpmVideoAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createModelChangesValidator(shard, campaignWithTypeByAdGroupId);

        return ListValidationBuilder.<ModelChanges<CpmVideoAdGroup>, Defect>of(modelChangesList)
                .checkEachBy(priceSalesValidator, When.valueIs(
                        changes -> campaignWithTypeByAdGroupId.get(changes.getId()).getType() == CampaignType.CPM_PRICE
                ))
                .getResult();
    }

    @Override
    public ValidationResult<List<CpmVideoAdGroup>, Defect> validateAdGroups(ClientId clientId,
                                                                            List<CpmVideoAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        var priceSalesValidator = priceSalesValidatorFactory.createBeanValidator();

        boolean isCpmVideoDisabled = featureService.isEnabledForClientId(clientId, FeatureName.DISABLE_CPM_VIDEO);

        return ListValidationBuilder.<CpmVideoAdGroup, Defect>of(adGroups)
                .checkEachBy(this::commonVideoAdGroupValidation)
                .checkEachBy(adGroup -> (campaignTypes.get(adGroup.getCampaignId()) == CampaignType.CPM_PRICE ) ?
                                validateCpmPriceVideoAdGroup(adGroup, priceSalesValidator) :
                                validateCpmVideoAdGroup(adGroup, isCpmVideoDisabled)
                        )
                .getResult();
    }

    private ValidationResult<CpmVideoAdGroup, Defect> validateCpmVideoAdGroup(CpmVideoAdGroup adGroup,
                                                                             boolean isCpmVideoDisabled) {
        ModelItemValidationBuilder<CpmVideoAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        UsersSegmentsValidationUtil.validateAdGroupWithUsersSegments(vb, adGroup.getType(), false);

        vb.item(AdGroup.TYPE)
                .check(fromPredicate(k -> !isCpmVideoDisabled, adGroupTypeNotSupported()));

        return vb.getResult();
    }

    private ValidationResult<CpmVideoAdGroup, Defect> validateCpmPriceVideoAdGroup(
            CpmVideoAdGroup adGroup,
            AdGroupPriceSalesValidator priceSalesValidator) {

        ModelItemValidationBuilder<CpmVideoAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.checkBy(priceSalesValidator);

        return vb.getResult();
    }

    private ValidationResult<CpmVideoAdGroup, Defect> commonVideoAdGroupValidation(
            CpmVideoAdGroup adGroup) {

        ModelItemValidationBuilder<CpmVideoAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroup.MINUS_KEYWORDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));
        vb.item(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<CpmVideoAdGroup>, Defect> validateAddAdGroups(
            ClientId clientId,
            List<CpmVideoAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        AdGroupPriceSalesAddValidator<CpmVideoAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createAddValidator(shard, campaignTypes, adGroups);

        return ListValidationBuilder.<CpmVideoAdGroup, Defect>of(adGroups)
                .checkEachBy(priceSalesValidator, When.valueIs(
                        adGroup -> campaignTypes.get(adGroup.getCampaignId()) == CampaignType.CPM_PRICE
                ))
                .getResult();
    }

    @Override
    public Class<CpmVideoAdGroup> getAdGroupClass() {
        return CpmVideoAdGroup.class;
    }

}
