package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CpmYndxFrontpageAdGroup;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.minusKeywordsNotAllowed;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;

@Component
@ParametersAreNonnullByDefault
public class CpmYndxFrontpageAdGroupValidation implements AdGroupTypeSpecificValidationService<CpmYndxFrontpageAdGroup> {

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory;

    public CpmYndxFrontpageAdGroupValidation(ShardHelper shardHelper,
                                             CampaignRepository campaignRepository,
                                             AdGroupPriceSalesValidatorFactory priceSalesValidatorFactory) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.priceSalesValidatorFactory = priceSalesValidatorFactory;
    }

    @Override
    public ValidationResult<List<CpmYndxFrontpageAdGroup>, Defect> validateAdGroups(
            ClientId clientId,
            List<CpmYndxFrontpageAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        var priceSalesValidator = priceSalesValidatorFactory.createBeanValidator();

        return ListValidationBuilder.<CpmYndxFrontpageAdGroup, Defect>of(adGroups)
                .checkEachBy(adGroup -> validateAdGroup(adGroup, campaignTypes, priceSalesValidator))
                .getResult();
    }

    private ValidationResult<CpmYndxFrontpageAdGroup, Defect> validateAdGroup(
            CpmYndxFrontpageAdGroup adGroup,
            Map<Long, CampaignType> campaignsType,
            AdGroupPriceSalesValidator priceSalesValidator) {
        var campaignType = campaignsType.get(adGroup.getCampaignId());

        ModelItemValidationBuilder<CpmYndxFrontpageAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroup.MINUS_KEYWORDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));
        vb.item(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));
        vb.item(CpmYndxFrontpageAdGroup.PRIORITY)
                .check(isNull(), When.isTrue(campaignType != CampaignType.CPM_PRICE));
        vb.checkBy(priceSalesValidator, When.isTrue(campaignType == CampaignType.CPM_PRICE));

        return vb.getResult();
    }

    @Override
    public ValidationResult<List<ModelChanges<CpmYndxFrontpageAdGroup>>, Defect> validateModelChanges(
            ClientId clientId,
            List<ModelChanges<CpmYndxFrontpageAdGroup>> modelChangesList) {
        int shard = shardHelper.getShardByClientId(clientId);
        var adGroupIds = listToSet(modelChangesList, ModelChanges::getId);
        var campaignWithTypeByAdGroupId =
                campaignRepository.getCampaignsWithTypeByAdGroupIds(shard, clientId, adGroupIds);

        AdGroupPriceSalesModelChangesValidator<CpmYndxFrontpageAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createModelChangesValidator(shard, campaignWithTypeByAdGroupId);

        return ListValidationBuilder.<ModelChanges<CpmYndxFrontpageAdGroup>, Defect>of(modelChangesList)
                .checkEachBy(priceSalesValidator, When.valueIs(
                        changes -> campaignWithTypeByAdGroupId.get(changes.getId()).getType() == CampaignType.CPM_PRICE
                ))
                .getResult();
    }

    @Override
    public ValidationResult<List<CpmYndxFrontpageAdGroup>, Defect> validateAddAdGroups(
            ClientId clientId,
            List<CpmYndxFrontpageAdGroup> adGroups) {
        int shard = shardHelper.getShardByClientId(clientId);
        var campaignIds = listToSet(adGroups, AdGroup::getCampaignId);
        var campaignTypes = campaignRepository.getCampaignsTypeMap(shard, campaignIds);


        AdGroupPriceSalesAddValidator<CpmYndxFrontpageAdGroup> priceSalesValidator =
                priceSalesValidatorFactory.createAddValidator(shard, campaignTypes, adGroups);

        return ListValidationBuilder.<CpmYndxFrontpageAdGroup, Defect>of(adGroups)
                .checkEachBy(priceSalesValidator, When.valueIs(
                        adGroup -> campaignTypes.get(adGroup.getCampaignId()) == CampaignType.CPM_PRICE
                ))
                .getResult();
    }

    @Override
    public Class<CpmYndxFrontpageAdGroup> getAdGroupClass() {
        return CpmYndxFrontpageAdGroup.class;
    }
}
