package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.DynamicFeedAdGroup;
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.model.FeedSimple;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.feed.validation.FeedDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Component
@ParametersAreNonnullByDefault
public class DynamicFeedAdGroupValidation extends BaseDynSmartAdGroupValidationService<DynamicFeedAdGroup> {

    private final FeedService feedService;

    public DynamicFeedAdGroupValidation(FeedService feedService) {
        this.feedService = feedService;
    }

    @Override
    public ValidationResult<List<ModelChanges<DynamicFeedAdGroup>>, Defect> validateModelChanges(ClientId clientId,
                                                                                                 List<ModelChanges<DynamicFeedAdGroup>> modelChangesList) {
        if (isEmpty(modelChangesList)) {
            return ValidationResult.success(modelChangesList);
        }
        List<Long> feedIds = mapList(modelChangesList, changes -> changes.getPropIfChanged(DynamicFeedAdGroup.FEED_ID));
        List<Long> existingFeedIds = mapList(feedService.getFeedsSimple(clientId,
                FeedQueryFilter.newBuilder().withFeedIds(feedIds).build()), FeedSimple::getId);

        ListValidationBuilder<ModelChanges<DynamicFeedAdGroup>, Defect> vb = ListValidationBuilder.of(modelChangesList);
        vb.checkEachBy(adGroupModelChangesValidator(existingFeedIds));
        return vb.getResult();
    }

    private Validator<ModelChanges<DynamicFeedAdGroup>, Defect> adGroupModelChangesValidator(List<Long> existingFeedIds) {
        return modelChanges -> {
            if (!modelChanges.isAnyPropChanged()) {
                return ValidationResult.success(modelChanges);
            }
            ItemValidationBuilder<ModelChanges<DynamicFeedAdGroup>, Defect> itemValidation =
                    ItemValidationBuilder.of(modelChanges);
            itemValidation
                    .item(modelChanges.getPropIfChanged(DynamicFeedAdGroup.FEED_ID), DynamicFeedAdGroup.FEED_ID.name())
                    .check(inSet(Set.copyOf(existingFeedIds)), FeedDefects.feedNotExist(), When.notNull());
            return itemValidation.getResult();
        };
    }

    @Override
    public ValidationResult<List<DynamicFeedAdGroup>, Defect> validateAddAdGroups(ClientId clientId,
                                                                                  List<DynamicFeedAdGroup> adGroups) {
        if (isEmpty(adGroups)) {
            ValidationResult.success(adGroups);
        }
        AdGroupWithFeedIdValidator<DynamicFeedAdGroup> feedIdValidator =
                new AdGroupWithFeedIdValidator.Builder<>(feedService, clientId, adGroups)
                        .build();
        ListValidationBuilder<DynamicFeedAdGroup, Defect> vb = ListValidationBuilder.of(adGroups);
        vb.checkEachBy(feedIdValidator);
        return vb.getResult();
    }

    @Override
    public Class<DynamicFeedAdGroup> getAdGroupClass() {
        return DynamicFeedAdGroup.class;
    }

}
