package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.internal.TemplateVariablesAgeValidator;
import ru.yandex.direct.core.entity.campaign.model.InternalCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.core.entity.internalads.InternalAdValidationUtil;
import ru.yandex.direct.core.entity.internalads.model.InternalAdsProduct;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.service.InternalAdsProductService;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.core.validation.defects.params.CollectionValuesDefectParams;
import ru.yandex.direct.core.validation.validators.ModelWithInternalImpressionRateValidator;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.minusKeywordsNotAllowed;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.result.ValidationResult.getInvalidItemsWithoutWarnings;

@Component
@ParametersAreNonnullByDefault
public class InternalAdGroupValidation extends BaseAdGroupTypeSpecificValidationService<InternalAdGroup> {

    public static final long MAX_LEVEL_VALUE = 1000L;

    private final BannerTypedRepository bannerTypedRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final TemplateInfoService templateInfoService;
    private final ShardHelper shardHelper;
    private final GeoTreeFactory geoTreeFactory;
    private final InternalAdsProductService internalAdsProductService;

    @Autowired
    public InternalAdGroupValidation(BannerTypedRepository bannerTypedRepository,
                                     TemplateInfoService templateInfoService,
                                     ShardHelper shardHelper,
                                     GeoTreeFactory geoTreeFactory,
                                     CampaignTypedRepository campaignTypedRepository,
                                     InternalAdsProductService internalAdsProductService) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.templateInfoService = templateInfoService;
        this.shardHelper = shardHelper;
        this.geoTreeFactory = geoTreeFactory;
        this.campaignTypedRepository = campaignTypedRepository;
        this.internalAdsProductService = internalAdsProductService;
    }


    @Override
    public ValidationResult<List<InternalAdGroup>, Defect> validateAdGroups(ClientId clientId,
                                                                            List<InternalAdGroup> adGroups) {
        var adGroupIds = listToSet(adGroups, InternalAdGroup::getId);
        int shard = shardHelper.getShardByClientId(clientId);

        var campaignMap = campaignTypedRepository.getTypedCampaignsMap(shard, mapList(adGroups,
                InternalAdGroup::getCampaignId));
        var internalAdsProduct = internalAdsProductService.getProduct(clientId);
        var banners = bannerTypedRepository.getBannersByGroupIds(shard, adGroupIds, InternalBanner.class);
        var bannersByAdGroupId = StreamEx.of(banners)
                .mapToEntry(InternalBanner::getAdGroupId, identity())
                .grouping();

        var templateIds = listToSet(banners, InternalBanner::getTemplateId);
        List<InternalTemplateInfo> templateInfoList = templateInfoService.getByTemplateIds(templateIds);

        return ListValidationBuilder.<InternalAdGroup, Defect>of(adGroups)
                .checkEachBy(adGroup ->
                        validateAdGroup(adGroup, (InternalCampaign) campaignMap.get(adGroup.getCampaignId()),
                                internalAdsProduct, bannersByAdGroupId.getOrDefault(adGroup.getId(), emptyList()),
                                templateInfoList))
                .getResult();
    }

    public ValidationResult<InternalAdGroup, Defect> validateAdGroup(InternalAdGroup adGroup,
                                                                     InternalCampaign campaign,
                                                                     InternalAdsProduct internalAdsProduct,
                                                                     List<InternalBanner> banners,
                                                                     List<InternalTemplateInfo> templateInfoList) {
        ModelItemValidationBuilder<InternalAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        vb.item(AdGroup.MINUS_KEYWORDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));
        vb.item(AdGroup.LIBRARY_MINUS_KEYWORDS_IDS)
                .check(fromPredicate(CollectionUtils::isEmpty, minusKeywordsNotAllowed()));

        vb.item(InternalAdGroup.LEVEL)
                .check(notLessThan(0L))
                .check(notGreaterThan(MAX_LEVEL_VALUE));

        List<Long> bannerWithCloseCounterVarIds = InternalAdValidationUtil
                .getBannerWithCloseCounterVarIds(banners, templateInfoList, Constants.CLOSE_BY_AD_GROUP_COUNTER_VALUE);
        vb.checkBy(new ModelWithInternalImpressionRateValidator<>(InternalAdGroup.RF, InternalAdGroup.RF_RESET,
                InternalAdGroup.MAX_CLICKS_COUNT, InternalAdGroup.MAX_CLICKS_PERIOD,
                InternalAdGroup.MAX_STOPS_COUNT, InternalAdGroup.MAX_STOPS_PERIOD,
                ifNotNull(campaign, InternalCampaign::getRfCloseByClick),
                bannerWithCloseCounterVarIds));

        vb.item(InternalAdGroup.FINISH_TIME)
                .checkByFunction(finishTime -> {
                    var startTime = InternalAdGroup.START_TIME.get(adGroup);
                    return (startTime != null && finishTime != null && finishTime.compareTo(startTime) <= 0) ?
                            AdGroupDefects.finishTimeShouldBeGreaterThanStartTime() : null;
                });
        vb.item(InternalAdGroup.GEO)
                .checkByFunction(geo -> validateAgeVariable(banners, geo, campaign, internalAdsProduct,
                        templateInfoList));
        return vb.getResult();
    }

    @Nullable
    public Defect validateAgeVariable(List<InternalBanner> banners, List<Long> adGroupGeo, InternalCampaign campaign,
                                      InternalAdsProduct internalAdsProduct,
                                      List<InternalTemplateInfo> templateInfoList) {
        var templateInfoMap = listToMap(templateInfoList, InternalTemplateInfo::getTemplateId);
        var campaignIsMobile = nvl(ifNotNull(campaign, InternalCampaign::getIsMobile), false);
        ListValidationBuilder<InternalBanner, Defect> vb = ListValidationBuilder.of(banners);
        vb.checkEachBy(bannerAgeVariableValidator(templateInfoMap, campaignIsMobile, adGroupGeo, internalAdsProduct));

        return vb.getResult().hasAnyErrors() ?
                requiredAgeVariable(vb.getResult()) : null;
    }

    private Defect<CollectionValuesDefectParams<Long>> requiredAgeVariable(
            ValidationResult<List<InternalBanner>, Defect> result) {
        List<Long> ids = mapList(getInvalidItemsWithoutWarnings(result), InternalBanner::getId);
        return AdGroupDefects.requiredAgeVariable(ids);
    }

    private Validator<InternalBanner, Defect> bannerAgeVariableValidator(
            Map<Long, InternalTemplateInfo> templateInfoMap,
            Boolean campaignIsMobile,
            List<Long> adGroupGeo,
            InternalAdsProduct internalAdsProduct) {
        return banner -> {
            ModelItemValidationBuilder<InternalBanner> vb = ModelItemValidationBuilder.of(banner);
            vb.item(InternalBanner.TEMPLATE_VARIABLES)
                    .checkBy(templateVariables ->
                            TemplateVariablesAgeValidator.templateVariablesAgeValidator(
                                    templateInfoMap.get(banner.getTemplateId()), internalAdsProduct,
                                    getGeoTree(), campaignIsMobile, adGroupGeo).apply(templateVariables));

            return vb.getResult();
        };
    }

    @Override
    public Class<InternalAdGroup> getAdGroupClass() {
        return InternalAdGroup.class;
    }

    @Override
    ValidationResult<InternalAdGroup, Defect> validateAdGroup(InternalAdGroup adGroup) {
        throw new UnsupportedOperationException("this method must not be called");
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getRussianGeoTree();
    }
}
