package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppService;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListItemValidator;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.advertisedAppLinkIsNotSet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.deviceTargetingIsNotSet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.invalidAppLinkFormat;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.minOsVersionIsNotSet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.networkTargetingIsNotSet;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.storeUrlMustBeTheSameAsUrlInMobileContentCampaign;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.storeUrlMustBeTheSameForAllGroupsInMobileContentCampaign;
import static ru.yandex.direct.core.validation.constraints.MobileContentConstraints.validAppStoreDomain;
import static ru.yandex.direct.core.validation.constraints.MobileContentConstraints.validAppStoreUrl;
import static ru.yandex.direct.core.validation.constraints.MobileContentConstraints.validOsVersion;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Component
@ParametersAreNonnullByDefault
public class MobileContentAdGroupValidation implements AdGroupTypeSpecificValidationService<MobileContentAdGroup> {
    private final FeatureService featureService;
    private final CampaignService campaignService;
    private final MobileAppService mobileAppService;

    MobileContentAdGroupValidation(FeatureService featureService,
                                   CampaignService campaignService,
                                   MobileAppService mobileAppService) {
        this.featureService = featureService;
        this.campaignService = campaignService;
        this.mobileAppService = mobileAppService;
    }

    @Override
    public ValidationResult<List<ModelChanges<MobileContentAdGroup>>, Defect> validateModelChanges(ClientId clientId,
                                                                                                   List<ModelChanges<MobileContentAdGroup>> modelChangesList) {
        return ValidationResult.success(modelChangesList);
    }

    @Override
    public ValidationResult<List<MobileContentAdGroup>, Defect> validateAdGroups(ClientId clientId,
                                                                                 List<MobileContentAdGroup> adGroups) {
        ListValidationBuilder<MobileContentAdGroup, Defect> vb = ListValidationBuilder.of(adGroups);
        vb.checkEachBy(this::validateAdGroup);

        Set<Long> campaignIds = StreamEx.of(adGroups)
                .map(AdGroup::getCampaignId)
                .nonNull()
                .toSet();
        Map<String, MobileAppStoreUrl> memoizationMap = new HashMap<>();
        Map<Long, MobileAppStoreUrl> campaignsStoreUrl = getCampaignsStoreUrl(clientId, campaignIds, memoizationMap);

        vb.checkEachBy(groupsInCampaignMustHaveSameStoreUrl(adGroups, memoizationMap));
        vb.checkEach(storeUrlMustBeEqualToUrlAttachedToCampaign(campaignsStoreUrl, memoizationMap));
        return vb.getResult();
    }

    public ValidationResult<MobileContentAdGroup, Defect> validateAdGroup(MobileContentAdGroup adGroup) {
        ModelItemValidationBuilder<MobileContentAdGroup> vb = ModelItemValidationBuilder.of(adGroup);

        MobileAppStoreUrl parsedUrl = MobileAppStoreUrlParser.parse(adGroup.getStoreUrl()).orElse(null);

        vb.item(MobileContentAdGroup.STORE_URL)
                .check(notNull(), advertisedAppLinkIsNotSet())
                .check(notBlank(), advertisedAppLinkIsNotSet())
                .check(validHref(), invalidAppLinkFormat(), When.isValid())
                .check(validAppStoreDomain(), When.isValid())
                .check(validAppStoreUrl(parsedUrl), When.isValid());

        vb.item(MobileContentAdGroup.MINIMAL_OPERATING_SYSTEM_VERSION)
                .check(notNull(), minOsVersionIsNotSet())
                .check(notBlank(), minOsVersionIsNotSet(), When.isValid())
                .check(validOsVersion(parsedUrl), When.isValid());

        vb.item(MobileContentAdGroup.DEVICE_TYPE_TARGETING)
                .check(notNull(), deviceTargetingIsNotSet())
                .check(notEmptyCollection(), deviceTargetingIsNotSet());

        vb.item(MobileContentAdGroup.NETWORK_TARGETING)
                .check(notNull(), networkTargetingIsNotSet())
                .check(notEmptyCollection(), networkTargetingIsNotSet());

        return vb.getResult();
    }

    @Override
    public Class<MobileContentAdGroup> getAdGroupClass() {
        return MobileContentAdGroup.class;
    }

    Constraint<MobileContentAdGroup, Defect> storeUrlMustBeEqualToUrlAttachedToCampaign(
            Map<Long, MobileAppStoreUrl> campaignStoreUrls, Map<String, MobileAppStoreUrl> memoizationMap) {
        return adGroup -> {
            if (adGroup == null || adGroup.getCampaignId() == null) {
                return null;
            }

            // проверяем ссылку только в добавляемых группах, меняться она не должна
            // https://st.yandex-team.ru/DIRECT-90430
            if (adGroup.getId() != null) {
                return null;
            }

            MobileAppStoreUrl campaignMobileAppStoreUrl = campaignStoreUrls.get(adGroup.getCampaignId());
            if (campaignMobileAppStoreUrl == null) {
                return null;
            }
            if (campaignMobileAppStoreUrl.equals(
                    MobileAppStoreUrlParser.parseMemoized(adGroup.getStoreUrl(), memoizationMap).orElse(null))) {
                return null;
            }
            return storeUrlMustBeTheSameAsUrlInMobileContentCampaign();
        };
    }

    private Map<Long, MobileAppStoreUrl> getCampaignsStoreUrl(ClientId clientId, Set<Long> campaignIds, Map<String,
            MobileAppStoreUrl> memoizationMap) {
        Map<Long, Long> campaignIdToMobileAppId = campaignService.getCampaignMobileAppId(clientId, campaignIds);
        Map<Long, String> mobileAppIdToStoreUrl =
                StreamEx.of(mobileAppService.getMobileApps(clientId, campaignIdToMobileAppId.values()))
                        .mapToEntry(MobileApp::getId, MobileApp::getStoreHref)
                        .toMap();
        return EntryStream.of(campaignIdToMobileAppId)
                .mapValues(mobileAppIdToStoreUrl::get)
                .nonNullValues()// для чужой кампании не найдётся урла
                //в базе должны лежать только валидные ссылки
                .mapValues(url -> MobileAppStoreUrlParser.parseStrictMemoized(url, memoizationMap))
                .toMap();
    }

    static ListItemValidator<MobileContentAdGroup, Defect> groupsInCampaignMustHaveSameStoreUrl(
            List<MobileContentAdGroup> adGroups, Map<String, MobileAppStoreUrl> memoizationMap) {
        Map<Long, Set<MobileAppStoreUrl>> campaignIdToUrls = StreamEx.of(adGroups)
                .filter(adGroup -> adGroup.getId() == null) // проверяем ссылку только в добавляемых группах
                .mapToEntry(AdGroup::getCampaignId, MobileContentAdGroup::getStoreUrl)
                .nonNullKeys() // если данные невалидны -- пропускаем, ругнётся на это другой валидатор
                .nonNullValues()
                .mapValues(url -> MobileAppStoreUrlParser.parseMemoized(url, memoizationMap))
                .flatMapValues(StreamEx::of) // если ссылки невалидны, пропускаем: ругнётся другой валидатор
                .grouping(Collectors.toSet());
        Set<Long> campaignsWithMoreThanOneUrl = EntryStream.of(campaignIdToUrls)
                .mapValues(Set::size)
                .filterValues(num -> num > 1)
                .keys()
                .toSet();
        Set<Integer> positionsOfGroupsWithDifferentStoreUrls = EntryStream.of(adGroups)
                .mapValues(AdGroup::getCampaignId)
                .nonNullValues()
                .filterValues(campaignsWithMoreThanOneUrl::contains)
                .keys()
                .toSet();

        return (index, item) -> {
            ValidationResult<MobileContentAdGroup, Defect> vr = ValidationResult.success(item);
            if (!positionsOfGroupsWithDifferentStoreUrls.contains(index)) {
                return vr;
            }
            // нужно добавлять ошибку на поле storeUrl
            ModelItemValidationBuilder<MobileContentAdGroup> vb = new ModelItemValidationBuilder<>(vr);
            vb.item(MobileContentAdGroup.STORE_URL)
                    .check(value -> storeUrlMustBeTheSameForAllGroupsInMobileContentCampaign());
            return vb.getResult();
        };
    }
}
