package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.model.FeedSimple;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.feed.validation.FeedDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentCreativeGeoToAdGroupGeo;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Component
@ParametersAreNonnullByDefault
public class PerformanceAdGroupValidation extends BaseDynSmartAdGroupValidationService<PerformanceAdGroup> {
    private final FeedService feedService;
    private final ClientGeoService clientGeoService;
    private final CreativeService creativeService;

    public PerformanceAdGroupValidation(FeedService feedService,
                                        ClientGeoService clientGeoService,
                                        CreativeService creativeService) {
        this.feedService = feedService;
        this.clientGeoService = clientGeoService;
        this.creativeService = creativeService;
    }

    @Override
    public ValidationResult<List<ModelChanges<PerformanceAdGroup>>, Defect> validateModelChanges(ClientId clientId,
                                                                                                 List<ModelChanges<PerformanceAdGroup>> modelChangesList) {
        if (isEmpty(modelChangesList)) {
            return ValidationResult.success(modelChangesList);
        }
        List<Long> adGroupIds = StreamEx.of(modelChangesList).map(ModelChanges::getId).toList();
        List<Long> feedIds = mapList(modelChangesList, it -> it.getPropIfChanged(PerformanceAdGroup.FEED_ID));
        List<Long> existingFeedIds = mapList(feedService.getFeedsSimple(clientId,
                FeedQueryFilter.newBuilder().withFeedIds(feedIds).build()), FeedSimple::getId);
        Map<Long, List<Creative>> creativesByPerformanceAdGroup =
                creativeService.getCreativesByPerformanceAdGroups(clientId, adGroupIds);
        ListValidationBuilder<ModelChanges<PerformanceAdGroup>, Defect> vb = ListValidationBuilder.of(modelChangesList);
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        vb.checkEachBy(adGroupModelChangesValidator(
                geoTree,
                creativesByPerformanceAdGroup,
                existingFeedIds
        ));
        return vb.getResult();
    }

    private Validator<ModelChanges<PerformanceAdGroup>, Defect> adGroupModelChangesValidator(
            GeoTree geoTree,
            Map<Long, List<Creative>> creativesByPerformanceAdGroup,
            List<Long> existingFeedIds) {
        return modelChanges -> {
            if (!modelChanges.isAnyPropChanged()) {
                return ValidationResult.success(modelChanges);
            }
            ItemValidationBuilder<ModelChanges<PerformanceAdGroup>, Defect> itemValidation =
                    ItemValidationBuilder.of(modelChanges);
            itemValidation
                    .item(modelChanges.getPropIfChanged(PerformanceAdGroup.FEED_ID), PerformanceAdGroup.FEED_ID.name())
                    .check(inSet(Set.copyOf(existingFeedIds)), FeedDefects.feedNotExist(), When.notNull());
            List<Creative> adGroupCreatives = creativesByPerformanceAdGroup.get(modelChanges.getId());
            itemValidation.item(modelChanges.getPropIfChanged(PerformanceAdGroup.GEO), PerformanceAdGroup.GEO.name())
                    .check(isAdGroupGeoCorrespondTo(geoTree, adGroupCreatives));
            return itemValidation.getResult();
        };
    }

    private Constraint<List<Long>, Defect> isAdGroupGeoCorrespondTo(GeoTree geoTree,
                                                                    Collection<Creative> adGroupCreatives) {
        if (isEmpty(adGroupCreatives)) {
            return value -> null;
        }
        Predicate<List<Long>> isConsistentGeoPredicate = adGroupGeo ->
        {
            Set<Long> adGroupCountries = geoTree.getModerationCountries(adGroupGeo);
            return BannerWithCreativeConstraints
                    .isConsistentCreativeGeoToAdGroupGeo(adGroupCreatives, adGroupCountries);
        };
        return fromPredicate(isConsistentGeoPredicate, inconsistentCreativeGeoToAdGroupGeo());
    }

    @Override
    public ValidationResult<List<PerformanceAdGroup>, Defect> validateAddAdGroups(ClientId clientId,
                                                                                  List<PerformanceAdGroup> adGroups) {
        if (isEmpty(adGroups)) {
            ValidationResult.success(adGroups);
        }

        AdGroupWithFeedIdValidator<PerformanceAdGroup> feedIdValidator =
                new AdGroupWithFeedIdValidator.Builder<>(feedService, clientId, adGroups)
                        .build();
        ListValidationBuilder<PerformanceAdGroup, Defect> vb = ListValidationBuilder.of(adGroups);
        vb.checkEachBy(feedIdValidator);
        return vb.getResult();
    }

    @Override
    public Class<PerformanceAdGroup> getAdGroupClass() {
        return PerformanceAdGroup.class;
    }

}
