package ru.yandex.direct.core.entity.adgroup.service.validation.types;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultPriority;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isSpecificPriority;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.CONTENT_CATEGORY;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.CONTENT_GENRE;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.retargetingConditionIsInvalidByDefaultAdGroup;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@ParametersAreNonnullByDefault
public class PriceSalesAdGroupContentCategoriesValidator implements DefaultValidator<List<Rule>> {
    public static final Set<GoalType> CONTENT_GENRE_AND_CATEGORY = Set.of(CONTENT_CATEGORY, CONTENT_GENRE);
    private final PriceSalesAdGroupContentCategoriesValidationData data;

    public PriceSalesAdGroupContentCategoriesValidator(PriceSalesAdGroupContentCategoriesValidationData validationData) {
        this.data = validationData;
    }

    @Override
    public ValidationResult<List<Rule>, Defect> apply(List<Rule> rules) {
        //сами категории валидируются в ContentCategoriesRetargetingConditionRulesValidator
        //В этом валидаторе только соответствие пакету аналогично RetargetingConditionsCpmPriceValidatorBase
        //И валидация специфичной группы об дефолтную
        boolean needDefaultAdGroup = data.getPricePackage().needDefaultAdGroup();
        return ListValidationBuilder.of(rules, Defect.class)
                .checkEachBy(this::validateRuleByPackage,
                        When.isTrue(!needDefaultAdGroup || isDefaultPriority(data.getPriority())))
                .checkEachBy(this::validateRuleForSpecificAdGroup,
                        When.isTrue(needDefaultAdGroup && isSpecificPriority(data.getPriority())))
                .check(Constraint.fromPredicateOfNullable(
                        r -> r != null || data.getDefAdGroupContentCategoriesRules() == null
                                || data.getDefAdGroupContentCategoriesRules().isEmpty(),
                        retargetingConditionIsInvalidByDefaultAdGroup()),
                        When.isTrue(needDefaultAdGroup && isSpecificPriority(data.getPriority())))
                .getResult();
    }

    private ValidationResult<Rule, Defect> validateRuleForSpecificAdGroup(Rule rule) {
        var goals = StreamEx.of(rule.getGoals())
                .filter(goal -> CONTENT_GENRE_AND_CATEGORY.contains(goal.getType()))
                .map(Goal::getId).toSet();
        Set<Long> defaultAdGroupGoals = data.getDefAdGroupContentCategoriesRules() == null ? emptySet() :
                StreamEx.of(data.getDefAdGroupContentCategoriesRules())
                        .flatMap(it -> it.getGoals().stream())
                        .map(Goal::getId).toSet();
        var vb = ModelItemValidationBuilder.of(rule);
        vb.list(Rule.GOALS)
                .check(Constraint.fromPredicate(
                        it -> defaultAdGroupGoals.isEmpty()
                                || !goals.isEmpty() && defaultAdGroupGoals.containsAll(goals),
                        retargetingConditionIsInvalidByDefaultAdGroup()));
        return vb.getResult();
    }

    private ValidationResult<Rule, Defect> validateRuleByPackage(Rule rule) {
        var fixedGoalIds = extractFixedGoalIds();
        var optionalGoalIds = extractOptionalGoalIds();
        var goals = StreamEx.of(rule.getGoals())
                .filter(goal -> CONTENT_GENRE_AND_CATEGORY.contains(goal.getType()))
                .map(Goal::getId).toSet();
        var vb = ModelItemValidationBuilder.of(rule);
        vb.list(Rule.GOALS)
                .check(Constraint.fromPredicate(
                        goalId -> goals.containsAll(fixedGoalIds) && optionalGoalIds.containsAll(goals),
                        invalidValue()));
        return vb.getResult();
    }

    private Set<Long> extractFixedGoalIds() {
        var pricePackage = data.getPricePackage();
        var set = new HashSet<Long>();
        if (pricePackage.getTargetingsFixed().getCryptaSegments() != null) {
            set.addAll(pricePackage.getTargetingsFixed().getCryptaSegments());
        }
        return StreamEx.of(set)//фильтруем только жанры и категории
                .filter(goalId -> CONTENT_GENRE_AND_CATEGORY.contains(Goal.computeType(goalId)))
                .toSet();
    }

    private Set<Long> extractOptionalGoalIds() {
        var set = extractFixedGoalIds();
        var pricePackage = data.getPricePackage();
        var retargetingCondition = pricePackage.getTargetingsCustom().getRetargetingCondition();
        if(retargetingCondition != null && retargetingCondition.getCryptaSegments() != null) {
            set.addAll(retargetingCondition.getCryptaSegments());
        }
        return set;
    }
}
