package ru.yandex.direct.core.entity.adgroupadditionaltargeting;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.StdDelegatingSerializer;
import com.fasterxml.jackson.databind.util.StdConverter;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.TimeAdGroupAdditionalTargeting;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.libs.timetarget.TimeTargetUtils;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.JsonUtils;

public class AdGroupAdditionalTargetingUtils {
    @SuppressWarnings({"unchecked", "rawtypes"})
    private static final ObjectMapper MAPPER = new ObjectMapper()
            .addMixIn(AdGroupAdditionalTargeting.class, FilteringMixIn.class)
            .configure(MapperFeature.SORT_PROPERTIES_ALPHABETICALLY, true)
            .configure(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS, true)
            .setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY)
            .registerModule(new SimpleModule().addSerializer(
                    new StdDelegatingSerializer(Set.class, new SetToListConverter())
            ))
            .registerModule(JsonUtils.createLocalDateModule());

    private static final List<TimeTarget> TIME_TARGET_24_7_USE_WORKING_WEEKEND_LIST =
            List.of(TimeTargetUtils.timeTarget24x7UseWorkingWeekend());

    public static Pair<Long, String> targetingIndexKey(AdGroupAdditionalTargeting targeting) {
        return Pair.of(targeting.getAdGroupId(), additionalTargetingDescriptor(targeting));
    }

    /**
     * Создаёт текстовый описатель таргетинга. Для одинаковых по смыслу таргетингов возвращается одинковый описатель.
     * Описатель не зависит от идентификатора таргетинга и группы.
     */
    public static String additionalTargetingDescriptor(AdGroupAdditionalTargeting targeting) {
        try {
            return targeting.getClass().getName() + MAPPER.writeValueAsString(targeting);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    /**
     * Нужно для значений, которые фронт нам присылает, но мы не хотим сохранять,
     * т. к. они никак не сужают аудиторию.
     */
    public static boolean isTargetingRelevantForStore(AdGroupAdditionalTargeting targeting) {
        var shouldIgnore = targeting instanceof TimeAdGroupAdditionalTargeting &&
                TIME_TARGET_24_7_USE_WORKING_WEEKEND_LIST.equals(
                        ((TimeAdGroupAdditionalTargeting) targeting).getValue()
                );
        return !shouldIgnore;
    }

    @SuppressWarnings("unused")
    abstract static class FilteringMixIn {
        @JsonIgnore
        private Long id;

        @JsonIgnore
        private Long adGroupId;
    }

    private static class SetToListConverter<T> extends StdConverter<Set<T>, List<String>> {
        @Override
        public List<String> convert(Set<T> value) {
            return value.stream()
                    .map(Checked.function(MAPPER::writeValueAsString))
                    .sorted()
                    .collect(Collectors.toList());
        }
    }
}
