package ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport.AdGroupAdditionalTargetingTypeSupportDispatcher;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupAdditionalTargetingsTargetingType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport.CommonAdditionalTargetingTypeSupport.ADGROUP_ADDITIONAL_TARGETINGS_MAPPER_FOR_COMMON_FIELDS;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupAdditionalTargetings.ADGROUP_ADDITIONAL_TARGETINGS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;

/**
 * Класс репозитория для работы разными таргетингами - классами-наследниками {@link AdGroupAdditionalTargeting}
 * Вызывает общие методы диспетчера {@link AdGroupAdditionalTargetingTypeSupportDispatcher} при сохранении объектов
 * в базу и чтении из базы.
 */
@Repository
@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingRepository {
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final Set<Field<?>> fieldsToRead;
    private final AdGroupAdditionalTargetingTypeSupportDispatcher adGroupAdditionalTargetingTypeSupportDispatcher;

    @Autowired
    public AdGroupAdditionalTargetingRepository(
            ShardHelper shardHelper,
            DslContextProvider dslContextProvider,
            AdGroupAdditionalTargetingTypeSupportDispatcher adGroupAdditionalTargetingTypeSupportDispatcher) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.fieldsToRead = new ImmutableSet.Builder<Field<?>>()
                .addAll(ADGROUP_ADDITIONAL_TARGETINGS_MAPPER_FOR_COMMON_FIELDS.getFieldsToRead())
                .add(ADGROUP_ADDITIONAL_TARGETINGS.VALUE).add(ADGROUP_ADDITIONAL_TARGETINGS.TARGETING_TYPE).build();
        this.adGroupAdditionalTargetingTypeSupportDispatcher = adGroupAdditionalTargetingTypeSupportDispatcher;
    }

    public List<Long> add(Configuration config, ClientId clientId, List<AdGroupAdditionalTargeting> targetings) {
        return add(config.dsl(), clientId, targetings);
    }

    public List<Long> add(int shard, ClientId clientId, List<AdGroupAdditionalTargeting> targetings) {
        return add(dslContextProvider.ppc(shard), clientId, targetings);
    }

    public List<Long> add(DSLContext context, ClientId clientId, List<AdGroupAdditionalTargeting> targetings) {
        List<Long> ids = shardHelper.generateAdGroupAdditionalTargetingIds(targetings.size());

        EntryStream.zip(targetings, ids).forKeyValue(AdGroupAdditionalTargeting::setId);

        context.transaction(configuration ->
                adGroupAdditionalTargetingTypeSupportDispatcher.addToDb(configuration.dsl(), clientId, targetings));

        return ids;
    }

    public List<AdGroupAdditionalTargeting> getByIds(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADGROUP_ADDITIONAL_TARGETINGS)
                .where(ADGROUP_ADDITIONAL_TARGETINGS.ID.in(ids))
                .fetch(adGroupAdditionalTargetingTypeSupportDispatcher::constructFromDb);
    }

    public List<AdGroupAdditionalTargeting> getByAdGroupId(int shard, Long adGroupId) {
        return getByAdGroupIds(shard, singleton(adGroupId));
    }

    public List<AdGroupAdditionalTargeting> getByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADGROUP_ADDITIONAL_TARGETINGS)
                .where(ADGROUP_ADDITIONAL_TARGETINGS.PID.in(adGroupIds))
                .fetch(adGroupAdditionalTargetingTypeSupportDispatcher::constructFromDb);
    }

    public List<AdGroupAdditionalTargeting> getByAdGroupIdsAndType(
            int shard, Collection<Long> adGroupIds, AdgroupAdditionalTargetingsTargetingType type) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return getByAdGroupIdsAndType(dslContext, adGroupIds, type);
    }

    public List<AdGroupAdditionalTargeting> getByAdGroupIdsAndType(DSLContext dslContext, Collection<Long> adGroupIds,
            AdgroupAdditionalTargetingsTargetingType type)
    {
        if (adGroupIds.isEmpty()) {
            return emptyList();
        }
        return dslContext
                .select(fieldsToRead)
                .from(ADGROUP_ADDITIONAL_TARGETINGS)
                .where(ADGROUP_ADDITIONAL_TARGETINGS.PID.in(adGroupIds)
                        .and(ADGROUP_ADDITIONAL_TARGETINGS.TARGETING_TYPE.eq(type)))
                .fetch(adGroupAdditionalTargetingTypeSupportDispatcher::constructFromDb);
    }

    public List<AdGroupAdditionalTargeting> getByClientIdsAndType(
            int shard, Collection<Long> clientIds, AdgroupAdditionalTargetingsTargetingType type) {
        if (clientIds.isEmpty()) {
            return emptyList();
        }

        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADGROUP_ADDITIONAL_TARGETINGS)
                .join(PHRASES)
                    .on(ADGROUP_ADDITIONAL_TARGETINGS.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS)
                    .on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(CAMPAIGNS.CLIENT_ID.in(clientIds)
                        .and(ADGROUP_ADDITIONAL_TARGETINGS.TARGETING_TYPE.eq(type)))
                .fetch(adGroupAdditionalTargetingTypeSupportDispatcher::constructFromDb);
    }

    public Set<Long> getAdGroupsByTargetingIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(ADGROUP_ADDITIONAL_TARGETINGS.PID)
                .from(ADGROUP_ADDITIONAL_TARGETINGS)
                .where(ADGROUP_ADDITIONAL_TARGETINGS.ID.in(ids))
                .fetchSet(ADGROUP_ADDITIONAL_TARGETINGS.PID);
    }

    public void deleteByAdGroupIds(Configuration config, Collection<Long> ids) {
        deleteByAdGroupIds(config.dsl(), ids);
    }

    public void deleteByAdGroupIds(DSLContext context, Collection<Long> ids) {
        context.deleteFrom(ADGROUP_ADDITIONAL_TARGETINGS)
                .where(ADGROUP_ADDITIONAL_TARGETINGS.PID.in(ids))
                .execute();
    }

    public void deleteByIds(int shard, Collection<Long> ids) {
        deleteByIds(dslContextProvider.ppc(shard), ids);
    }

    public void deleteByIds(DSLContext context, Collection<Long> ids) {
        context.deleteFrom(ADGROUP_ADDITIONAL_TARGETINGS)
                .where(ADGROUP_ADDITIONAL_TARGETINGS.ID.in(ids))
                .execute();
    }
}
