package ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.configuration.AdGroupAdditionalTargetingTypeAccessor;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.configuration.AdGroupAdditionalTargetingWithParametricValueAccessor;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.configuration.AdGroupAdditionalTargetingsConfigurationProvider;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.MobileInstalledAppsAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.ShowDatesAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.TimeAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport.valuefieldmapper.ShowDatesValueFieldMapper;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport.valuefieldmapper.TimeTargetFieldMapper;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport.valuefieldmapper.ValueFieldMapper;
import ru.yandex.direct.dbschema.ppc.enums.AdgroupAdditionalTargetingsTargetingType;
import ru.yandex.direct.dbschema.ppc.tables.records.AdgroupAdditionalTargetingsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport.valuefieldmapper.ValueFieldMappers.fromTypeReference;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupAdditionalTargetings.ADGROUP_ADDITIONAL_TARGETINGS;

/**
 * Компонент, который реализует правильное сохранение и извлечение из базы
 * объектов дополнительных таргетингов разных типов. Разные AdGroupAdditionalTargetingTypeSupport
 * в большинстве случаев должны быть скрыты за ним,
 * а он сам делегирует управление в нужный класс при работе с базой.
 */
@Component
@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingTypeSupportDispatcher {

    private final Map<Class<? extends AdGroupAdditionalTargeting>,
            AdGroupAdditionalTargetingTypeSupport<AdGroupAdditionalTargeting>>
            typeSupportMap;
    private final Map<AdgroupAdditionalTargetingsTargetingType,
            AdGroupAdditionalTargetingTypeSupport<AdGroupAdditionalTargeting>>
            typeSupportMapByDBType;

    @Autowired
    public AdGroupAdditionalTargetingTypeSupportDispatcher(
            MobileInstalledAppsAdditionalTargetingTypeSupport mobileInstalledAppsAdGroupAdditionalTargetingTypeSupport) {

        Map<Class<?>, AdGroupAdditionalTargetingTypeSupport<AdGroupAdditionalTargeting>> specificTypeSupports =
                Map.ofEntries(Map.entry(
                        MobileInstalledAppsAdGroupAdditionalTargeting.class,
                        upCast(mobileInstalledAppsAdGroupAdditionalTargetingTypeSupport))
                );

        var specificFieldMappers = Map.of(
                ShowDatesAdGroupAdditionalTargeting.class, new ShowDatesValueFieldMapper(),
                TimeAdGroupAdditionalTargeting.class, new TimeTargetFieldMapper()
        );

        List<AdGroupAdditionalTargetingTypeSupport<AdGroupAdditionalTargeting>> typeSupports =
                StreamEx.of(AdGroupAdditionalTargetingsConfigurationProvider.getAccessors())
                        .map(accessor -> {
                            var targetingClass = accessor.getTargetingClass();

                            if (specificTypeSupports.containsKey(targetingClass)) {
                                return specificTypeSupports.get(targetingClass);
                            }

                            return upCast(toTypeSupport(accessor, specificFieldMappers));
                        })
                        .toList();

        this.typeSupportMap = typeSupports.stream()
                .collect(toMap(AdGroupAdditionalTargetingTypeSupport::getTargetingClass, identity()));

        this.typeSupportMapByDBType = StreamEx.of(typeSupports)
                .mapToEntry(AdGroupAdditionalTargetingTypeSupport::getDbType, identity())
                .toMap();
    }

    private <T extends AdGroupAdditionalTargeting, V extends Collection<?>>
    AdGroupAdditionalTargetingTypeSupport<? extends AdGroupAdditionalTargeting>
    toTypeSupport(AdGroupAdditionalTargetingTypeAccessor<T> accessor,
                  Map<Class<? extends AdGroupAdditionalTargeting>, ? extends ValueFieldMapper<?>> specificFieldMappers) {

        if (accessor instanceof AdGroupAdditionalTargetingWithParametricValueAccessor) {
            var parametricValueAccessor =
                    (AdGroupAdditionalTargetingWithParametricValueAccessor<T, V>) accessor;

            @SuppressWarnings("unchecked")
            ValueFieldMapper<V> fieldMapper =
                    (ValueFieldMapper<V>) specificFieldMappers.get(accessor.getTargetingClass());

            if (fieldMapper == null) {
                fieldMapper = fromTypeReference(JsonUtils.getTypeFactory().constructParametricType(
                        parametricValueAccessor.getValueClass(), parametricValueAccessor.getValueParameterClass()));
            }

            return new CommonAdditionalTargetingTypeSupport<>(
                    parametricValueAccessor.getTargetingClass(),
                    parametricValueAccessor::newTargeting,
                    parametricValueAccessor.getTargetingType(),
                    parametricValueAccessor.getValueModelProperty(),
                    fieldMapper);
        }

        return new CommonAdditionalTargetingTypeSupport<>(
                accessor.getTargetingClass(),
                accessor::newTargeting,
                accessor.getTargetingType());
    }

    @SuppressWarnings("unchecked")
    private <T extends AdGroupAdditionalTargeting> AdGroupAdditionalTargetingTypeSupport<AdGroupAdditionalTargeting>
    upCast(AdGroupAdditionalTargetingTypeSupport<T> typeSupport) {
        return (AdGroupAdditionalTargetingTypeSupport<AdGroupAdditionalTargeting>) typeSupport;
    }

    public void addToDb(DSLContext dslContext, ClientId clientId, List<AdGroupAdditionalTargeting> targetings) {
        InsertHelper<AdgroupAdditionalTargetingsRecord> insertHelper =
                new InsertHelper<>(dslContext, ADGROUP_ADDITIONAL_TARGETINGS);

        targetings.stream()
                .collect(groupingBy(AdGroupAdditionalTargeting::getClass, toList()))
                .forEach((clazz, list) -> typeSupportMap.get(clazz)
                        .addToDbHelper(insertHelper, dslContext, clientId, list));
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(ADGROUP_ADDITIONAL_TARGETINGS.VALUE, MySQLDSL.values(ADGROUP_ADDITIONAL_TARGETINGS.VALUE));
        }
        insertHelper.executeIfRecordsAdded();
    }

    private AdGroupAdditionalTargetingTypeSupport<AdGroupAdditionalTargeting> getTypeSupportForDbRecord(Record record) {
        AdgroupAdditionalTargetingsTargetingType dbType = record.get(ADGROUP_ADDITIONAL_TARGETINGS.TARGETING_TYPE);
        return checkNotNull(typeSupportMapByDBType.get(dbType));
    }

    public AdGroupAdditionalTargeting constructFromDb(Record record) {
        return getTypeSupportForDbRecord(record).constructFromDb(record);
    }
}
