package ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.typesupport.valuefieldmapper;

import java.io.IOException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;

import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.json.LocalDateDeserializer;

@ParametersAreNonnullByDefault
public class ShowDatesValueFieldMapper implements ValueFieldMapper<Set<LocalDate>> {
    /**
     * Указанный формат хранения таргетитнга требуется для его работы на стороне БК.
     * Правильным способом его передачи было бы совершение соответствующего преобразования в транспорте
     * и разделение логики передачи данных во внешний сервис от логики их хранения, но в таком подходе
     * придётся парсить внутреннюю структуру AdditionalTargeting'ов и разделять способы их обработки
     * в транспорте в БК в зависимости от типа таргетинга
     */
    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("_yyyyMMdd______");

    private static final class ShowDateSerializer extends LocalDateSerializer {
        ShowDateSerializer() {
            super(FORMATTER);
        }
    }

    private static final class ShowDateDeserializer extends LocalDateDeserializer {
        @Override
        public LocalDate deserialize(JsonParser p, DeserializationContext ctxt)
                throws IOException, JsonProcessingException {
            return LocalDate.parse(p.getText(), FORMATTER);
        }
    }

    private static final ObjectMapper MAPPER = new ObjectMapper().registerModule(
            new JavaTimeModule()
                    .addSerializer(LocalDate.class, new ShowDateSerializer())
                    .addDeserializer(LocalDate.class, new ShowDateDeserializer()))
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    private static final JavaType TYPE_REFERENCE = JsonUtils.getTypeFactory()
            .constructCollectionType(Set.class, LocalDate.class);

    @Override
    public Set<LocalDate> fromJson(String json) {
        try {
            return MAPPER.readValue(json, TYPE_REFERENCE);
        } catch (IOException e) {
            throw new IllegalArgumentException("can not deserialize object from json", e);
        }
    }

    @Override
    public String toJson(Set<LocalDate> obj) {
        try {
            return MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }
}
