package ru.yandex.direct.core.entity.adgroupadditionaltargeting.service;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.AdGroupAdditionalTargetingUtils;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.AdGroupAdditionalTargetingRepository;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.validation.AdGroupAdditionalTargetingsValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.OperationsUtils;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.AdGroupAdditionalTargetingUtils.targetingIndexKey;
import static ru.yandex.direct.utils.FunctionalUtils.batchDispatch;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingFindOrCreateOperation
        extends AbstractAddOperation<AdGroupAdditionalTargeting, AddedModelId> {
    private final AdGroupAdditionalTargetingRepository additionalTargetingRepository;
    private final AdGroupAdditionalTargetingService adGroupAdditionalTargetingService;
    private final AdGroupAdditionalTargetingsValidationService adGroupAdditionalTargetingsValidationService;
    private final AdGroupRepository adGroupRepository;
    private final boolean partOfComplexOperation;
    private final int shard;
    private final ClientId clientId;

    AdGroupAdditionalTargetingFindOrCreateOperation(
            List<AdGroupAdditionalTargeting> targetings,
            AdGroupAdditionalTargetingRepository additionalTargetingRepository,
            AdGroupAdditionalTargetingService adGroupAdditionalTargetingService,
            AdGroupAdditionalTargetingsValidationService adGroupAdditionalTargetingsValidationService,
            AdGroupRepository adGroupRepository, boolean partOfComplexOperation, int shard, ClientId clientId) {
        super(Applicability.PARTIAL, targetings);
        this.additionalTargetingRepository = additionalTargetingRepository;
        this.adGroupAdditionalTargetingService = adGroupAdditionalTargetingService;
        this.adGroupAdditionalTargetingsValidationService = adGroupAdditionalTargetingsValidationService;
        this.adGroupRepository = adGroupRepository;
        this.partOfComplexOperation = partOfComplexOperation;
        this.shard = shard;
        this.clientId = clientId;
    }

    @Override
    protected void validate(ValidationResult<List<AdGroupAdditionalTargeting>, Defect> preValidationResult) {
        new ItemValidationBuilder<>(preValidationResult)
                .checkBy(targetings -> adGroupAdditionalTargetingsValidationService
                        .validate(targetings, partOfComplexOperation));
    }

    @Override
    protected Map<Integer, AddedModelId> execute(Map<Integer, AdGroupAdditionalTargeting> validModelsMapToApply) {
        return OperationsUtils.applyForMapValues(validModelsMapToApply, this::execute);
    }

    private List<AddedModelId> execute(List<AdGroupAdditionalTargeting> validTargetings) {
        Set<Long> adGroupIds = listToSet(validTargetings, AdGroupAdditionalTargeting::getAdGroupId);

        // targeting index key -> targeting ID (см. targetingIndexKey(...))
        Map<Pair<Long, String>, Long> existing = calcExistingTargeting(adGroupIds);

        List<AddedModelId> result = batchDispatch(validTargetings,
                targeting -> existing.containsKey(targetingIndexKey(targeting)),
                targetings -> mapList(targetings, t -> AddedModelId.ofExisting(existing.get(targetingIndexKey(t)))),
                targetings -> mapList(additionalTargetingRepository.add(shard, clientId, targetings),
                        AddedModelId::ofNew)
        );

        Set<Long> affectedAdGroupIds = calcAffectedAdGroups(result);
        adGroupRepository.actualizeAdGroupsOnChildModification(shard, affectedAdGroupIds);
        return result;
    }

    /**
     * Найти группы, в которые происходило добавление таргетингов
     */
    private Set<Long> calcAffectedAdGroups(List<AddedModelId> result) {
        Set<Long> addedTargetingIds = StreamEx.of(result)
                .filter(AddedModelId::isAdded)
                .map(AddedModelId::getId)
                .toSet();
        return additionalTargetingRepository.getAdGroupsByTargetingIds(shard, addedTargetingIds);
    }

    /**
     * Создать индекс существующих на группах таргетингов
     */
    private Map<Pair<Long, String>, Long> calcExistingTargeting(Set<Long> adGroupIds) {
        List<AdGroupAdditionalTargeting> existingTargetings =
                adGroupAdditionalTargetingService.getTargetingsByAdGroupIds(clientId, adGroupIds);
        return StreamEx.of(existingTargetings)
                .mapToEntry(AdGroupAdditionalTargetingUtils::targetingIndexKey, AdGroupAdditionalTargeting::getId)
                .toMap();
    }
}
