package ru.yandex.direct.core.entity.adgroupadditionaltargeting.service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer.RequestSource;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.AdGroupAdditionalTargetingUtils;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.TimeAdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.AdGroupAdditionalTargetingRepository;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.container.AdGroupAdditionalTargetingGroupsOperationContainer;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.PartiallyApplicableOperation;
import ru.yandex.direct.operation.creator.GroupOperationCreator;
import ru.yandex.direct.operation.creator.ValidationOperationCreator;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Predicate.not;
import static ru.yandex.direct.core.entity.adgroup.container.InternalAdGroupOperationContainer.RequestSource.EXCEL;
import static ru.yandex.direct.utils.FunctionalUtils.extractSubList;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;

/**
 * Приводит доп. таргетинги на группе в соответствие с тем, что запрошено. Таргетинги которых ещё не было на группе
 * будут добавлены; таргетинги, которые есть на группе, но уже не нужны -- удалены.
 */
@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingGroupsUpdateOperation implements PartiallyApplicableOperation<List<AddedModelId>> {
    private final AdGroupAdditionalTargetingService additionalTargetingService;
    private final AdGroupAdditionalTargetingRepository additionalTargetingRepository;
    private final PartiallyApplicableOperation<List<AddedModelId>> findOrCreateGroupOperation;
    private final AdGroupRepository adGroupRepository;

    private final List<Long> adGroupIdForTargetingGroups;
    private final ClientId clientId;
    private final int shard;
    private final List<List<AdGroupAdditionalTargeting>> targetingGroups;

    private boolean prepared;
    private boolean executed;
    private final RequestSource requestSource;

    // Нужно ли оставлять старые значения указанных таргетингов, если они не пришли в запросе на обновление группы
    // при запросе от RequestSource
    private static final Map<RequestSource, Set<Class<? extends AdGroupAdditionalTargeting>>> TARGETINGS_WHICH_NOT_DELETE_IF_NEED =
            Map.of(
                    EXCEL, Set.of(TimeAdGroupAdditionalTargeting.class)
            );

    public AdGroupAdditionalTargetingGroupsUpdateOperation(
            AdGroupAdditionalTargetingService additionalTargetingService,
            AdGroupAdditionalTargetingRepository additionalTargetingRepository,
            AdGroupAdditionalTargetingFindOrCreateOperationFactory findOrCreateOperationFactory,
            AdGroupRepository adGroupRepository,
            AdGroupAdditionalTargetingGroupsOperationContainer operationContainer,
            int shard) {
        this.targetingGroups = operationContainer.getTargetingGroups();
        this.adGroupIdForTargetingGroups = operationContainer.getAdGroupIdForTargetingGroups();
        this.additionalTargetingService = additionalTargetingService;
        this.additionalTargetingRepository = additionalTargetingRepository;
        this.adGroupRepository = adGroupRepository;
        this.clientId = operationContainer.getClientId();
        this.shard = shard;
        this.requestSource = operationContainer.getRequestSource();

        EntryStream.zip(adGroupIdForTargetingGroups, targetingGroups)
                .flatMapValues(Collection::stream)
                .forKeyValue((adGroupId, targeting) -> targeting.setAdGroupId(adGroupId));
        var findOrCreateGroupOpCreator = new ValidationOperationCreator<>(new GroupOperationCreator<>(
                (List<AdGroupAdditionalTargeting> targetings) -> findOrCreateOperationFactory.newInstance(
                        targetings, true, clientId)),
                findOrCreateOperationFactory.getValidatorSupplier().getGroupValidator());
        findOrCreateGroupOperation = findOrCreateGroupOpCreator.create(targetingGroups);
    }

    @Override
    public Optional<MassResult<List<AddedModelId>>> prepare() {
        checkState(!prepared, "prepare() can be called only once");
        prepared = true;
        return findOrCreateGroupOperation.prepare();
    }

    @Override
    public Set<Integer> getValidElementIndexes() {
        checkState(prepared, "prepare() must be called before getValidElementIndexes()");
        return findOrCreateGroupOperation.getValidElementIndexes();
    }

    @Override
    public MassResult<List<AddedModelId>> apply(Set<Integer> elementIndexesToApply) {
        checkApplyOrCancelPreconditions();
        checkArgument(Sets.difference(elementIndexesToApply, getValidElementIndexes()).isEmpty(),
                "elementIndexesToApply contains indexes of invalid elements");
        executed = true;

        Set<Pair<Long, String>> desired = calcDesiredTargetings(
                extractSubList(targetingGroups, List.copyOf(elementIndexesToApply)));

        List<Long> mentionedAdGroupIds =
                extractSubList(adGroupIdForTargetingGroups, List.copyOf(elementIndexesToApply));
        List<AdGroupAdditionalTargeting> existingTargetings =
                additionalTargetingService.getTargetingsByAdGroupIds(clientId, mentionedAdGroupIds);

        // adGroupId -> набор ID тарегетингов, подлежащих удалению
        Map<Long, Set<Long>> toDelete = calcToDelete(existingTargetings, desired);

        var additionalTargetingIdsToDelete = flatMapToSet(toDelete.entrySet(), Map.Entry::getValue);
        additionalTargetingRepository.deleteByIds(shard, additionalTargetingIdsToDelete);

        MassResult<List<AddedModelId>> result = findOrCreateGroupOperation.apply(elementIndexesToApply);

        var affectedAdGroupIds = toDelete.keySet();
        adGroupRepository.actualizeAdGroupsOnChildModification(shard, affectedAdGroupIds);

        return result;
    }

    @Override
    public MassResult<List<AddedModelId>> apply() {
        return apply(getValidElementIndexes());
    }

    @Override
    public MassResult<List<AddedModelId>> cancel() {
        checkApplyOrCancelPreconditions();
        executed = true;
        return findOrCreateGroupOperation.cancel();
    }

    @Override
    public Optional<MassResult<List<AddedModelId>>> getResult() {
        return findOrCreateGroupOperation.getResult();
    }

    private Map<Long, Set<Long>> calcToDelete(
            List<AdGroupAdditionalTargeting> existingTargetings, Set<Pair<Long, String>> desired) {
        return StreamEx.of(existingTargetings)
                .remove(targeting -> isNotDeletableTargeting(targeting, desired))
                .mapToEntry(AdGroupAdditionalTargetingUtils::targetingIndexKey, AdGroupAdditionalTargeting::getId)
                .filterKeys(not(desired::contains))
                .mapKeys(Pair::getLeft) // adGroupId
                .grouping(Collectors.toSet());
    }

    private boolean isNotDeletableTargeting(AdGroupAdditionalTargeting targeting, Set<Pair<Long, String>> desired) {
        if (TARGETINGS_WHICH_NOT_DELETE_IF_NEED.getOrDefault(requestSource, Collections.emptySet()).contains(targeting.getClass())) {
            checkState(!desired.contains(AdGroupAdditionalTargetingUtils.targetingIndexKey(targeting)), "desired must" +
                    " not contains targetings which are in TARGETINGS_WHICH_NOT_DELETE_IF_NEED");
            return true;
        }

        return false;
    }

    private Set<Pair<Long, String>> calcDesiredTargetings(List<List<AdGroupAdditionalTargeting>> targetingGroups) {
        return StreamEx.of(targetingGroups)
                .flatMap(Collection::stream)
                .map(AdGroupAdditionalTargetingUtils::targetingIndexKey)
                .toSet();
    }

    private void checkApplyOrCancelPreconditions() {
        checkState(prepared, "prepare() must be called before apply() or cancel()");
        checkState(!executed, "apply() or cancel() can be called only once");
    }
}
