package ru.yandex.direct.core.entity.adgroupadditionaltargeting.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.AdGroupAdditionalTargetingRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;

@Service
@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingService implements EntityService<AdGroupAdditionalTargeting, Long> {
    private final AdGroupAdditionalTargetingRepository repository;
    private final AdGroupAdditionalTargetingsAddOperationFactory addOperationFactory;
    private final AdGroupAdditionalTargetingsDeleteOperationFactory deleteOperationFactory;
    private final ShardHelper shardHelper;

    public AdGroupAdditionalTargetingService(
            AdGroupAdditionalTargetingRepository repository,
            AdGroupAdditionalTargetingsAddOperationFactory addOperationFactory,
            AdGroupAdditionalTargetingsDeleteOperationFactory deleteOperationFactory,
            ShardHelper shardHelper) {
        this.repository = repository;
        this.addOperationFactory = addOperationFactory;
        this.deleteOperationFactory = deleteOperationFactory;
        this.shardHelper = shardHelper;
    }

    public List<AdGroupAdditionalTargeting> getTargetings(ClientId clientId, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyList();
        }

        return repository.getByIds(shardHelper.getShardByClientIdStrictly(clientId), ids);
    }

    public Map<Long, List<AdGroupAdditionalTargeting>> getTargetingMapByAdGroupIds(
            ClientId clientId, Collection<Long> adGroupIds) {
        var targetings = getTargetingsByAdGroupIds(clientId, adGroupIds);
        return StreamEx.of(targetings)
                .mapToEntry(AdGroupAdditionalTargeting::getAdGroupId, identity())
                .grouping();
    }

    public List<AdGroupAdditionalTargeting> getTargetingsByAdGroupIds(ClientId clientId, Collection<Long> adGroupIds) {
        return repository.getByAdGroupIds(shardHelper.getShardByClientIdStrictly(clientId), adGroupIds);
    }

    public MassResult<Long> deleteTargetings(ClientId clientId, List<Long> ids) {
        return deleteOperationFactory
                .newInstance(Applicability.FULL, ids, shardHelper.getShardByClientIdStrictly(clientId))
                .prepareAndApply();
    }

    public AdGroupAdditionalTargetingsAddOperation createAddOperation(
            Applicability applicability,
            List<AdGroupAdditionalTargeting> targetings,
            ClientId clientId) {
        return addOperationFactory.newInstance(
                applicability,
                false,
                targetings,
                clientId
        );
    }

    @Override
    public List<AdGroupAdditionalTargeting> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        return getTargetings(clientId, ids);
    }

    @Override
    public MassResult<Long> add(
            ClientId clientId, Long operatorUid, List<AdGroupAdditionalTargeting> entities,
            Applicability applicability) {
        return createAddOperation(applicability, entities, clientId).prepareAndApply();
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                List<AdGroupAdditionalTargeting> entities, Applicability applicability) {
        return add(copyContainer.getClientIdTo(), copyContainer.getOperatorUid(), entities, applicability);
    }

}
