package ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.Multiset;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.configuration.AdGroupAdditionalTargetingValueAccessor;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.flatMapping;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toCollection;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.duplicatedObject;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.configuration.AdGroupAdditionalTargetingsConfigurationProvider.getCollectionValueAccessorByTargeting;
import static ru.yandex.direct.core.entity.adgroupadditionaltargeting.configuration.AdGroupAdditionalTargetingsConfigurationProvider.getValueAccessorByTargeting;
import static ru.yandex.direct.validation.Predicates.each;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

/**
 * Сервис валидации дополнительных таргетингов на наличие дублирующихся элеметов в поле value
 */
@Service
@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingsDuplicateValidationService {

    public ValidationResult<List<AdGroupAdditionalTargeting>, Defect> validateTargetings(
            List<AdGroupAdditionalTargeting> targetings) {
        ListValidationBuilder<AdGroupAdditionalTargeting, Defect> vb =
                ListValidationBuilder.of(targetings);

        Map<Class<?>, Multiset<Object>> indexOfDuplicatesByClass = createIndexOfDuplicatesByClass(targetings);

        vb.checkEachBy(t -> targetingValidationDispatcher(t, indexOfDuplicatesByClass.get(t.getClass())),
                When.valueIs(t -> indexOfDuplicatesByClass.containsKey(t.getClass())));

        return vb.getResult();
    }

    private Map<Class<?>, Multiset<Object>> createIndexOfDuplicatesByClass(
            List<AdGroupAdditionalTargeting> targetings) {
        return targetings.stream()
                .filter(t -> getCollectionValueAccessorByTargeting(t) != null)
                .collect(groupingBy(Object::getClass,
                        flatMapping(t -> Objects.requireNonNull(getCollectionValueAccessorByTargeting(t)).getValue(t).stream(),
                                toCollection(HashMultiset::create))));
    }

    private <T extends AdGroupAdditionalTargeting> ValidationResult<T, Defect> targetingValidationDispatcher(
            T targeting, Multiset<Object> indexOfDuplicates) {
        AdGroupAdditionalTargetingValueAccessor<T, Set<Object>> targetingAccessorWithSetValue =
                getValueAccessorByTargeting(targeting, Set.class);

        if (targetingAccessorWithSetValue != null) {
            return validateTargetingWithSetValue(
                    targeting, targetingAccessorWithSetValue, indexOfDuplicates);
        }

        AdGroupAdditionalTargetingValueAccessor<T, List<Object>> targetingAccessorWithListValue =
                getValueAccessorByTargeting(targeting, List.class);

        if (targetingAccessorWithListValue != null) {
            return validateTargetingWithListValue(
                    targeting, targetingAccessorWithListValue, indexOfDuplicates);
        }

        throw new UnsupportedOperationException("unexpected targeting value representation");
    }

    private <T extends AdGroupAdditionalTargeting> ValidationResult<T, Defect> validateTargetingWithSetValue(
            T targeting,
            AdGroupAdditionalTargetingValueAccessor<T, Set<Object>> targetingAccessor,
            Multiset<Object> indexOfDuplicates
    ) {
        ModelItemValidationBuilder<T> vb = ModelItemValidationBuilder.of(targeting);

        vb.item(targetingAccessor.getValueModelProperty())
                .check(doesNotHaveDuplicateInSet(indexOfDuplicates));

        return vb.getResult();
    }

    private static <R> Constraint<Set<R>, Defect> doesNotHaveDuplicateInSet(Multiset<Object> indexOfDuplicates) {
        return fromPredicate(each(e -> indexOfDuplicates.count(e) == 1), duplicatedObject());
    }

    private <T extends AdGroupAdditionalTargeting> ValidationResult<T, Defect> validateTargetingWithListValue(
            T targeting,
            AdGroupAdditionalTargetingValueAccessor<T, List<Object>> targetingAccessor,
            Multiset<Object> indexOfDuplicates
    ) {
        ModelItemValidationBuilder<T> vb = ModelItemValidationBuilder.of(targeting);

        vb.list(targetingAccessor.getValueModelProperty())
                .checkEach(doesNotHaveDuplicateInList(indexOfDuplicates));

        return vb.getResult();
    }

    private static <R> Constraint<R, Defect> doesNotHaveDuplicateInList(Multiset<Object> indexOfDuplicates) {
        return fromPredicate(e -> indexOfDuplicates.count(e) == 1, duplicatedObject());
    }
}
