package ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.validation;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.validation.types.AdGroupAdditionalTargetingTypeSpecificValidationProvider;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

/**
 * Сервис валидации дополнительных таргетингов.
 * <p>
 * Валидация здесь ответствена только за содержимое таргетинга. Более высокоуровневые
 * проверки (например, наличие дубликатов среди таргетингов) должны совершаться
 * на уровне API, использующего этот сервис.
 */
@Service
@ParametersAreNonnullByDefault
public class AdGroupAdditionalTargetingsValidationService {
    private final AdGroupAdditionalTargetingTypeSpecificValidationProvider validationProvider;
    private final AdGroupAdditionalTargetingsDuplicateValidationService duplicateValidation;

    @Autowired
    public AdGroupAdditionalTargetingsValidationService(
            AdGroupAdditionalTargetingTypeSpecificValidationProvider validationProvider,
            AdGroupAdditionalTargetingsDuplicateValidationService duplicateValidation) {
        this.validationProvider = validationProvider;
        this.duplicateValidation = duplicateValidation;
    }

    public ValidationResult<List<AdGroupAdditionalTargeting>, Defect> preValidateByGroup(
            List<AdGroupAdditionalTargeting> targetings) {
        ListValidationBuilder<AdGroupAdditionalTargeting, Defect> vb = ListValidationBuilder.of(targetings);
        vb.checkBy(duplicateValidation::validateTargetings);

        return vb.getResult();
    }

    public ValidationResult<List<AdGroupAdditionalTargeting>, Defect> validate(
            List<AdGroupAdditionalTargeting> targetings, boolean partOfComplexOperation) {
        ListValidationBuilder<AdGroupAdditionalTargeting, Defect> vb = ListValidationBuilder.of(targetings);
        vb.checkEachBy(t -> validateTargeting(t, partOfComplexOperation));
        vb.checkBy(validationProvider::validateTargetings, When.isValid());

        return vb.getResult();
    }

    private ValidationResult<AdGroupAdditionalTargeting, Defect> validateTargeting(AdGroupAdditionalTargeting targeting,
                                                                                   boolean partOfComplexOperation) {
        ModelItemValidationBuilder<AdGroupAdditionalTargeting> vb = ModelItemValidationBuilder.of(targeting);

        if (!partOfComplexOperation) {
            // в комплексной операции группы валидируются отдельно
            vb.item(AdGroupAdditionalTargeting.AD_GROUP_ID)
                    .check(notNull())
                    .check(validId());
        }
        vb.item(AdGroupAdditionalTargeting.JOIN_TYPE)
                .check(notNull());
        vb.item(AdGroupAdditionalTargeting.TARGETING_MODE)
                .check(notNull());
        return vb.getResult();
    }
}
