package ru.yandex.direct.core.entity.adgroupadditionaltargeting.service.validation.types

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingJoinType.ANY
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargetingMode.TARGETING
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.TimeAdGroupAdditionalTargeting
import ru.yandex.direct.libs.timetarget.HoursCoef
import ru.yandex.direct.libs.timetarget.TimeTarget
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints.collectionSize
import ru.yandex.direct.validation.constraint.CommonConstraints.isEqual
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder

class TimeTargetingValidation : AdGroupAdditionalTargetingTypeSpecificValidationService<TimeAdGroupAdditionalTargeting> {
    override fun validateAdGroupAdditionalTargeting(
        targeting: TimeAdGroupAdditionalTargeting
    ): ValidationResult<TimeAdGroupAdditionalTargeting, Defect<Any>> {

        val vb = ModelItemValidationBuilder.of(targeting)
        vb.item(TimeAdGroupAdditionalTargeting.VALUE)
            .check(notNull())
            .check(collectionSize(1, 1))
            // требую не только профиль, а реальные коэффициенты!
            .check(timeTargetsNotEmpty(), When.isValid())
            // только вкл/выкл, без коэффициентов
            .check(timeTargetsWithoutMultipliers(), When.isValid())

        // TimeTarget работает только в режиме TARGETING
        vb.item(TimeAdGroupAdditionalTargeting.TARGETING_MODE)
            .check(isEqual(TARGETING, invalidValue()))
        // фиксируем значения joinType т.к. возможен только один TimeTarget
        vb.item(TimeAdGroupAdditionalTargeting.JOIN_TYPE)
            .check(isEqual(ANY, invalidValue()))
        return vb.result
    }

    override fun getTargetingClass() = TimeAdGroupAdditionalTargeting::class.java
}

private fun timeTargetsNotEmpty(): Constraint<List<TimeTarget>, Defect<*>?> {
    return fromPredicate({ allTimeTargetsNotEmpty(it) }, invalidValue())
}

private fun timeTargetsWithoutMultipliers(): Constraint<List<TimeTarget>, Defect<*>?> {
    return fromPredicate({ allTimeTargetsWithoutMultipliers(it) }, invalidValue())
}

private fun allTimeTargetsNotEmpty(timeTargets: Collection<TimeTarget>): Boolean {
    return timeTargets.all { it.isNotEmpty() }
}

private fun allTimeTargetsWithoutMultipliers(timeTargets: Collection<TimeTarget>): Boolean {
    return timeTargets.all { !it.hasMultipliers() }
}

private fun TimeTarget.isNotEmpty(): Boolean {
    return weekdayCoefs.isNotEmpty()
}

private fun TimeTarget.hasMultipliers(): Boolean {
    return weekdayCoefs.any { (_, hoursCoef) -> hoursCoef.hasMultipliers() }
}

private fun HoursCoef.hasMultipliers(): Boolean {
    return (0..23).map { getCoefForHour(it) }.any { it != 0 && it != 100 }
}
