package ru.yandex.direct.core.entity.agency.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.Field;
import org.jooq.Record2;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.agency.model.AgencyAdditionalCurrency;
import ru.yandex.direct.core.entity.application.model.AgencyOptions;
import ru.yandex.direct.core.entity.user.model.AgencyLimRep;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppc.tables.records.AgencyCurrenciesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.rbac.RbacAgencyLimRepType;

import static ru.yandex.direct.dbschema.ppc.Tables.AGENCY_LIM_REP_CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.AGENCY_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS_AGENCY;
import static ru.yandex.direct.dbschema.ppc.tables.AgencyCurrencies.AGENCY_CURRENCIES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class AgencyRepository {

    private final Collection<Field<?>> allFields;
    private final Collection<Field<?>> agencyLimRepsFields;
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<AgencyAdditionalCurrency> additionalCurrencyJooqMapper;
    private final JooqMapperWithSupplier<AgencyLimRep> agencyLimRepJooqMapper;

    @Autowired
    public AgencyRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = Objects.requireNonNull(
                dslContextProvider, "ppcDslContextProvider");

        additionalCurrencyJooqMapper = JooqMapperWithSupplierBuilder.builder(AgencyAdditionalCurrency::new)
                .map(property(AgencyAdditionalCurrency.CLIENT_ID, AGENCY_CURRENCIES.CLIENT_ID))
                .map(convertibleProperty(AgencyAdditionalCurrency.CURRENCY_CODE, AGENCY_CURRENCIES.CURRENCY,
                        AgencyAdditionalCurrencyMapping::currencyFromDb,
                        AgencyAdditionalCurrencyMapping::currencyToDb))
                .map(property(AgencyAdditionalCurrency.EXPIRATION_DATE, AGENCY_CURRENCIES.EXPIRATION_DATE))
                .map(property(AgencyAdditionalCurrency.LAST_CHANGE, AGENCY_CURRENCIES.LAST_CHANGE))
                .build();
        allFields = additionalCurrencyJooqMapper.getFieldsToRead();

        agencyLimRepJooqMapper = JooqMapperWithSupplierBuilder.builder(AgencyLimRep::new)
                .map(property(AgencyLimRep.UID, USERS_AGENCY.UID))
                .map(convertibleProperty(AgencyLimRep.REP_TYPE, USERS_AGENCY.LIM_REP_TYPE,
                        RbacAgencyLimRepType::fromSource, RbacAgencyLimRepType::toSource))
                .map(property(AgencyLimRep.GROUP_ID, USERS_AGENCY.GROUP_ID))
                .build();

        agencyLimRepsFields = agencyLimRepJooqMapper.getFieldsToRead();
    }

    /**
     * Возвращает список дополнительных актуальных на данный момент валют для агентства (См. Agency.pm::agency_currencies)
     * <p>
     * Список заполняется нотификациями со стороны Баланса
     */
    public Set<CurrencyCode> getAdditionalCurrencies(int shard, ClientId agencyClientId) {
        return dslContextProvider.ppc(shard)
                .select(AGENCY_CURRENCIES.CURRENCY)
                .from(AGENCY_CURRENCIES)
                .where(
                        AGENCY_CURRENCIES.CLIENT_ID.eq(agencyClientId.asLong())
                                .and(AGENCY_CURRENCIES.EXPIRATION_DATE.ge(DSL.currentLocalDate())))
                .fetchSet(r -> CurrencyCode.valueOf(r.value1().getLiteral()));
    }

    /**
     * Получить список всех дополнительных валют агентств с переданными идентификаторами
     *
     * @param shard           шард
     * @param agencyClientIds список clientId агентств
     */
    public List<AgencyAdditionalCurrency> getAllAdditionalCurrencies(int shard, Collection<Long> agencyClientIds) {
        if (agencyClientIds.isEmpty()) {
            return Collections.emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(allFields)
                .from(AGENCY_CURRENCIES)
                .where(AGENCY_CURRENCIES.CLIENT_ID.in(agencyClientIds))
                .fetch()
                .map(additionalCurrencyJooqMapper::fromDb);
    }

    /**
     * Вставить в базу данных новые записи с дополнительными валютами агентств. В случае, если запись для конкретного
     * агентства и валюты уже есть, перезаписать дату истечения валюты и таимстамп последнего изменения
     */
    public int addAdditionalCurrencies(int shard, Collection<AgencyAdditionalCurrency> additionalCurrencyList) {
        InsertHelper<AgencyCurrenciesRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), AGENCY_CURRENCIES);
        insertHelper.addAll(additionalCurrencyJooqMapper, additionalCurrencyList);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(AGENCY_CURRENCIES.EXPIRATION_DATE, MySQLDSL.values(AGENCY_CURRENCIES.EXPIRATION_DATE))
                    .set(AGENCY_CURRENCIES.LAST_CHANGE, LocalDateTime.now());
        }

        return insertHelper.executeIfRecordsAdded();
    }

    public AgencyOptions getAgencyOptions(int shard, ClientId clientId) {
        Record2<Long, Long> row = dslContextProvider.ppc(shard)
                .select(AGENCY_OPTIONS.ALLOW_CLIENTS_WITHOUT_WALLET, AGENCY_OPTIONS.DEFAULT_CLIENTS_WITH_WALLET)
                .from(AGENCY_OPTIONS)
                .where(AGENCY_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .fetchOne();
        if (row == null) {
            return new AgencyOptions();
        }
        return new AgencyOptions(row.get(AGENCY_OPTIONS.ALLOW_CLIENTS_WITHOUT_WALLET) != 0L,
                row.get(AGENCY_OPTIONS.DEFAULT_CLIENTS_WITH_WALLET) != 0);
    }

    /**
     * Обновить поле agency_options.deal_notification_email
     *
     * @param shard    шард
     * @param clientId clientID агентства
     * @param email    email для записи в deal_notification_email
     */
    public void setDealNotificationEmail(int shard, ClientId clientId, String email) {
        dslContextProvider.ppc(shard)
                .update(AGENCY_OPTIONS)
                .set(AGENCY_OPTIONS.DEAL_NOTIFICATION_EMAIL, email)
                .where(AGENCY_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .execute();
    }

    /**
     * Получить agency_options.deal_notification_email для агентства
     *
     * @param shard    шард
     * @param clientId clientId агентства
     */
    @Nullable
    public String getDealNotificationEmail(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(AGENCY_OPTIONS.DEAL_NOTIFICATION_EMAIL)
                .from(AGENCY_OPTIONS)
                .where(AGENCY_OPTIONS.CLIENT_ID.eq(clientId.asLong()))
                .fetchOne(AGENCY_OPTIONS.DEAL_NOTIFICATION_EMAIL);
    }

    /**
     * Возвращает типы ограниченных представителей агентства по uid'ам
     *
     * @param shard шард
     * @param limRepUids uid'ы ограниченных представителей
     */
    public Map<Long, AgencyLimRep> getAgencyLimReps(int shard, Collection<Long> limRepUids) {
        return dslContextProvider.ppc(shard)
                .select(agencyLimRepsFields)
                .from(USERS_AGENCY)
                .where(USERS_AGENCY.UID.in(new HashSet<>(limRepUids)))
                .fetchMap(USERS_AGENCY.UID, agencyLimRepJooqMapper::fromDb);
    }

    /**
     * Возвращает мапу идентификаторов клиентов в список ограниченных представителей агентства, к которым привязан указанный клиент
     *
     * @param shard шард
     * @param clientIds  список идентификаторов клиентов
     */
    public Map<Long, Set<Long>> getAgencyLimRepUidsByClients(int shard, Collection<Long> clientIds) {
         var limRepUidsByClients = dslContextProvider.ppc(shard)
                .select(AGENCY_LIM_REP_CLIENTS.CLIENT_ID, AGENCY_LIM_REP_CLIENTS.AGENCY_UID)
                .from(AGENCY_LIM_REP_CLIENTS)
                .where(AGENCY_LIM_REP_CLIENTS.CLIENT_ID.in(new HashSet<>(clientIds)))
                .fetchGroups(AGENCY_LIM_REP_CLIENTS.CLIENT_ID, AGENCY_LIM_REP_CLIENTS.AGENCY_UID);
         return EntryStream.of(limRepUidsByClients).mapValues(Set::copyOf).toMap();
    }
}
