package ru.yandex.direct.core.entity.agency.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.agency.model.AgencyAdditionalCurrency;
import ru.yandex.direct.core.entity.agency.repository.AgencyRepository;
import ru.yandex.direct.core.entity.application.model.AgencyOptions;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.service.CurrencyDictCache;
import ru.yandex.direct.core.entity.user.model.AgencyLimRep;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.Region;

import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;

/**
 * Сервис для работы с агентствами
 */
@Service
@ParametersAreNonnullByDefault
public class AgencyService {
    private final AgencyRepository agencyRepository;
    private final ClientService clientService;
    private final CurrencyDictCache currencyDictCache;
    private final ShardHelper shardHelper;
    private final RbacService rbacService;

    @Autowired
    public AgencyService(
            AgencyRepository agencyRepository,
            ClientService clientService,
            ShardHelper shardHelper,
            CurrencyDictCache currencyDictCache,
            RbacService rbacService) {
        this.agencyRepository = Objects.requireNonNull(agencyRepository, "agencyRepository");
        this.clientService = Objects.requireNonNull(clientService, "clientService");
        this.currencyDictCache = Objects.requireNonNull(currencyDictCache, "currencyService");
        this.shardHelper = Objects.requireNonNull(shardHelper, "shardHelper");
        this.rbacService = rbacService;
    }

    /**
     * Вернуть валюты в которых может платить агентство с точки зрения Direct-а (См. Agency.pm::_get_agency_allowed_pay_currencies)
     * <p>
     * Используется для агентств, которые платят в YND_FIXED
     */
    private Set<CurrencyCode> getAllowedToPayCurrenciesForYndFixesAgencies(ClientId agencyClientId) {
        Multimap<Long, CurrencyCode> allowedForPayCurrencies = clientService.getAllowedForPayCurrencies(agencyClientId);

        // ВАЖНО: Для новых агентств баланс возвращает полный список валют по всем странам,
        // который соотвествующей job-ой переносится в таблицу client_firm_country_currency,
        // поэтому если разрешать агентству выбирать для субклиента любую из этих валют, то возможна
        // ситуация описанная ниже
        // Оригинальный комментарий из perl-а:
        // Все варианты == агентство не платило и может в чём угодны создавать субклиентов. Но после первой оплаты
        // останется только валюта оплаты. Чтобы сам себе не выстрелил в ногу, создав клиента, за которого не может
        // платить — не даём клиентам с таким набором выбирать реальные валюты
        if (allowedForPayCurrencies.size() == currencyDictCache.getAnyCountryCurrencyCount().getAgencyCurrencyCount()) {
            return emptySet();
        }

        return allowedForPayCurrencies.values()
                .stream()
                .distinct()
                .collect(toSet());
    }

    /**
     * Вернуть допустимые для агентства валюты с точки зрения Direct-а (См. Agency.pm::get_agency_allowed_currencies_hash)
     *
     * @param agencyClientId Идентификатор агентства
     */
    public Set<CurrencyCode> getAllowedCurrencies(ClientId agencyClientId) {
        CurrencyCode currency = clientService.getWorkCurrency(agencyClientId).getCode();

        Set<CurrencyCode> currencies = new HashSet<>();

        if (currency == CurrencyCode.YND_FIXED) {
            Optional<Long> countryRegionId = clientService.getCountryRegionIdByClientId(agencyClientId);
            if (countryRegionId.isPresent() && countryRegionId.get().equals(Region.BY_REGION_ID)) {
                // DIRECT-56867, для Белорусских фишечных агентств - прибиваем гвоздями BYN
                currencies.add(CurrencyCode.BYN);
            } else {
                // уе-шные агентства могут создавать субклиентов в доступных им для оплаты валютах
                currencies.addAll(getAllowedToPayCurrenciesForYndFixesAgencies(agencyClientId));
            }
        } else {
            currencies.add(currency);
        }

        // Агентствам по доп.соглашению могут быть на ограниченный период доступны дополнительные валюты
        currencies.addAll(
                agencyRepository.getAdditionalCurrencies(
                        clientService.getShardByClientIdStrictly(agencyClientId),
                        agencyClientId));

        return currencies;
    }

    /**
     * Получить список всех дополнительных валют агентства с переданным идентификатором
     */
    public List<AgencyAdditionalCurrency> getAllAdditionalCurrencies(Long agencyClientId) {
        return getAllAdditionalCurrencies(Collections.singleton(agencyClientId));
    }

    /**
     * Получить список всех дополнительных валют агентств с переданными идентификаторами
     */
    public List<AgencyAdditionalCurrency> getAllAdditionalCurrencies(Collection<Long> agencyClientIds) {
        return shardHelper.groupByShard(agencyClientIds, ShardKey.CLIENT_ID)
                .stream()
                .map(e -> agencyRepository.getAllAdditionalCurrencies(e.getKey(), e.getValue()))
                .flatMap(List::stream)
                .collect(Collectors.toList());
    }

    /**
     * Вставить в базу данных новые записи с дополнительными валютами агентств. В случае, если запись для конкретного
     * агентства и валюты уже есть, перезаписать дату истечения валюты и таимстамп последнего изменения
     */
    public int addAdditionalCurrencies(Collection<AgencyAdditionalCurrency> additionalCurrencyList) {
        additionalCurrencyList.forEach(currency -> currency.withLastChange(LocalDateTime.now()));
        return shardHelper
                .groupByShard(additionalCurrencyList, ShardKey.CLIENT_ID, AgencyAdditionalCurrency::getClientId)
                .stream()
                .mapToInt(e -> agencyRepository.addAdditionalCurrencies(e.getKey(), e.getValue()))
                .sum();
    }

    public AgencyOptions getAgencyOptions(ClientId agencyId) {
        int shard = shardHelper.getShardByClientId(agencyId);
        return agencyRepository.getAgencyOptions(shard, agencyId);
    }

    /**
     * Получить логины шефов субклиентов для шеф uid агентства и отфильтровать по loginPart
     */
    public List<String> getSubClientLogins(Long chiefUid, long limit, @Nullable String loginPart) {
        List<Long> allSubClients = rbacService.getAgencySubclients(chiefUid);
        List<Long> chiefSubClients = rbacService.getChiefSubclients(allSubClients);
        List<String> subLogins =
                StreamEx.of(shardHelper.getLoginsByUids(chiefSubClients)).toFlatList(Function.identity());
        StreamEx<String> subLoginsStream = StreamEx.of(subLogins);
        if (isNotEmpty(loginPart)) {
            subLoginsStream = subLoginsStream.filter(t -> t.contains(loginPart));
        }
        return subLoginsStream
                .distinct()
                .sorted()
                .limit(limit)
                .toList();
    }

    /**
     * Обновить email агентства для нотификаций по сделкам
     *
     * @param clientId clientID агентства
     * @param email    email для нотификаций по сделкам
     */
    public void setDealNotificationEmail(ClientId clientId, String email) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        agencyRepository.setDealNotificationEmail(shard, clientId, email);
    }

    /**
     * Получить email агентства для нотификаций по сделкам
     *
     * @param clientId clientId агентства
     */
    @Nullable
    public String getDealNotificationEmail(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return agencyRepository.getDealNotificationEmail(shard, clientId);
    }

    /**
     * Получаем мапу сетов ограниченных представителей по идентификаторам клиентов
     *
     * @param clientIds идентификаторы клиентов
     */
    public Map<Long, Set<AgencyLimRep>> getAgencyLimRepsByClientIds(Collection<Long> clientIds) {
        var limRepUidsByClientIds = shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .stream()
                .mapKeyValue(agencyRepository::getAgencyLimRepUidsByClients)
                .flatMap(EntryStream::of)
                .toMap(Map.Entry::getKey, Map.Entry::getValue);
        var limRepUids = limRepUidsByClientIds.values().stream()
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
        var limRepTypesByUids = getAgencyLimRepsByUids(limRepUids);
        return limRepUidsByClientIds.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().stream()
                        .filter(limRepTypesByUids::containsKey)
                        .map(limRepTypesByUids::get)
                        .collect(Collectors.toSet())));
    }

    /**
     * Получаем объекты ограниченных представителей по их uid'ам
     *
     * @param agencyLimRepUids uid'ы ограниченных представителей
     */
    public Map<Long, AgencyLimRep> getAgencyLimRepsByUids(Collection<Long> agencyLimRepUids) {
        return shardHelper.groupByShard(new HashSet<>(agencyLimRepUids), ShardKey.UID)
                .stream()
                .mapKeyValue(agencyRepository::getAgencyLimReps)
                .flatMap(EntryStream::of)
                .toMap(Map.Entry::getKey, Map.Entry::getValue);
    }

}
