package ru.yandex.direct.core.entity.agencyofflinereport.repository;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.jooq.Condition;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.agencyofflinereport.container.AgencyOfflineReportArgs;
import ru.yandex.direct.core.entity.agencyofflinereport.model.AgencyOfflineReport;
import ru.yandex.direct.core.entity.agencyofflinereport.model.AgencyOfflineReportState;
import ru.yandex.direct.dbschema.ppc.enums.AgencyOfflineReportsReportState;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.compressedMediumblobJsonProperty;
import static ru.yandex.direct.dbschema.ppc.tables.AgencyOfflineReports.AGENCY_OFFLINE_REPORTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class AgencyOfflineReportRepository {

    private static final int DELETE_CHUNK_SIZE = 500;

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<AgencyOfflineReport> jooqMapper;

    @Autowired
    public AgencyOfflineReportRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(AgencyOfflineReport::new)
                .map(property(AgencyOfflineReport.REPORT_ID, AGENCY_OFFLINE_REPORTS.AGENCY_OFFLINE_REPORT_ID))
                .map(property(AgencyOfflineReport.UID, AGENCY_OFFLINE_REPORTS.UID))
                .map(clientIdProperty(AgencyOfflineReport.AGENCY_CLIENT_ID, AGENCY_OFFLINE_REPORTS.AGENCY_CLIENT_ID))
                .map(compressedMediumblobJsonProperty(AgencyOfflineReport.ARGS, AGENCY_OFFLINE_REPORTS.ARGS,
                        AgencyOfflineReportArgs.class))
                .map(property(AgencyOfflineReport.SCHEDULED_AT, AGENCY_OFFLINE_REPORTS.SCHEDULED_AT))
                .map(convertibleProperty(AgencyOfflineReport.REPORT_STATE, AGENCY_OFFLINE_REPORTS.REPORT_STATE,
                        AgencyOfflineReportState::fromSource, AgencyOfflineReportState::toSource))
                .map(property(AgencyOfflineReport.REPORT_URL, AGENCY_OFFLINE_REPORTS.REPORT_URL))
                .build();
    }

    /**
     * Получить параметры отчета по его id
     *
     * @param shard    шард
     * @param reportId идентификатор отчета
     * @return параметры отчета или {@code null}
     */
    public AgencyOfflineReport getAgencyOfflineReport(int shard, long reportId) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(AGENCY_OFFLINE_REPORTS)
                .where(AGENCY_OFFLINE_REPORTS.AGENCY_OFFLINE_REPORT_ID.eq(reportId))
                .fetchOne(jooqMapper::fromDb);
    }

    /**
     * Добавить отчет в список
     *
     * @param shard         шард
     * @param offlineReport параметры отчета
     */
    public void addAgencyOfflineReport(int shard, AgencyOfflineReport offlineReport) {
        new InsertHelper<>(dslContextProvider.ppc(shard), AGENCY_OFFLINE_REPORTS)
                .add(jooqMapper, offlineReport)
                .execute();
    }

    /**
     * Получить список оффлайн-отчетов по агентству.
     * Можно ограничить выборку отчетами, заказанными конкретным оператором (передав значение {@code reportOwnerUid}).
     * Отчеты возвращаются упорядоченными по убыванию их id.
     *
     * @param shard          шард агентства
     * @param agencyClientId id агентства
     * @param reportOwnerUid uid владельца отчета для фильтрации, опционально
     * @return список отчетов
     */
    public List<AgencyOfflineReport> getAgencyOfflineReports(int shard, ClientId agencyClientId,
                                                             @Nullable Long reportOwnerUid) {
        Condition operatorCondition;
        if (reportOwnerUid != null) {
            operatorCondition = AGENCY_OFFLINE_REPORTS.UID.eq(reportOwnerUid);
        } else {
            operatorCondition = MySQLDSL.trueCondition();
        }
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(AGENCY_OFFLINE_REPORTS)
                .where(AGENCY_OFFLINE_REPORTS.AGENCY_CLIENT_ID.eq(agencyClientId.asLong()).and(operatorCondition))
                .orderBy(AGENCY_OFFLINE_REPORTS.AGENCY_OFFLINE_REPORT_ID.desc())
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Перевести отчет из статуса "обрабатывается" в статус "готово", сохранить ссылку
     *
     * @param shard     шард
     * @param reportId  идентификатор отчета
     * @param reportUrl ссылка на построенный отчет
     */
    public void markProcessingReportAsReady(int shard, long reportId, String reportUrl) {
        dslContextProvider.ppc(shard)
                .update(AGENCY_OFFLINE_REPORTS)
                .set(AGENCY_OFFLINE_REPORTS.REPORT_STATE, AgencyOfflineReportsReportState.Ready)
                .set(AGENCY_OFFLINE_REPORTS.REPORT_URL, reportUrl)
                .where(AGENCY_OFFLINE_REPORTS.AGENCY_OFFLINE_REPORT_ID.eq(reportId)
                        .and(AGENCY_OFFLINE_REPORTS.REPORT_STATE.eq(AgencyOfflineReportsReportState.Processing)))
                .execute();
    }

    /**
     * Перевести отчет из статуса "новый" в статус "обрабатывается"
     *
     * @param shard    шард
     * @param reportId идентификатор отчета
     */
    public void markNewReportAsProcessing(int shard, long reportId) {
        dslContextProvider.ppc(shard)
                .update(AGENCY_OFFLINE_REPORTS)
                .set(AGENCY_OFFLINE_REPORTS.REPORT_STATE, AgencyOfflineReportsReportState.Processing)
                .where(AGENCY_OFFLINE_REPORTS.AGENCY_OFFLINE_REPORT_ID.eq(reportId)
                        .and(AGENCY_OFFLINE_REPORTS.REPORT_STATE.eq(AgencyOfflineReportsReportState.New)))
                .execute();
    }

    /**
     * Перевести отчет в состояние "ошибка обработки", затереть ссылку на готовый отчет
     *
     * @param shard    шард
     * @param reportId идентификатор отчета
     */
    public void markReportFailed(int shard, long reportId) {
        dslContextProvider.ppc(shard)
                .update(AGENCY_OFFLINE_REPORTS)
                .set(AGENCY_OFFLINE_REPORTS.REPORT_STATE, AgencyOfflineReportsReportState.Error)
                .set(AGENCY_OFFLINE_REPORTS.REPORT_URL, (String) null)
                .where(AGENCY_OFFLINE_REPORTS.AGENCY_OFFLINE_REPORT_ID.eq(reportId))
                .execute();
    }

    /**
     * Получить список отчетов, добавленных до {@code borderDateTime}
     *
     * @param shard          шард
     * @param borderDateTime граничное время
     * @return список отчетов
     * @implNote делает FULLSCAN таблицы, не
     */
    public List<AgencyOfflineReport> getOutdatedReports(int shard, LocalDateTime borderDateTime) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(AGENCY_OFFLINE_REPORTS)
                .where(AGENCY_OFFLINE_REPORTS.SCHEDULED_AT.lt(borderDateTime))
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Удалить отчеты
     *
     * @param shard     шард
     * @param reportIds id отчетов для удаления
     * @return количество удаленных строк
     * @implNote чанкует запросы к базе по {@link #DELETE_CHUNK_SIZE} штук
     */
    public int deleteReports(int shard, Iterable<Long> reportIds) {
        int result = 0;
        for (List<Long> chunk : Iterables.partition(reportIds, DELETE_CHUNK_SIZE)) {
            result += dslContextProvider.ppc(shard)
                    .deleteFrom(AGENCY_OFFLINE_REPORTS)
                    .where(AGENCY_OFFLINE_REPORTS.AGENCY_OFFLINE_REPORT_ID.in(chunk))
                    .execute();
        }
        return result;
    }
}
