package ru.yandex.direct.core.entity.agencyofflinereport.service;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;

@ParametersAreNonnullByDefault
@Service
public class AgencyOfflineReportParametersService {
    private static final Logger logger = LoggerFactory.getLogger(AgencyOfflineReportParametersService.class);
    private static final DateTimeFormatter MAX_DATE_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE;

    /**
     * Дата начиная с которой есть исходные данные для рассчетов
     */
    public static final LocalDate MIN_REPORT_DATE = LocalDate.of(2017, 1, 1);

    /**
     * Имя свойства, в котором хранится максимально-допустимая дата отчета.
     */
    public static final String MAX_REPORT_DATE_PROP_NAME = "AGENCY_OFFLINE_REPORT_MAX_DATE";

    /**
     * Дата, начиная с которой есть исходные данные для рассчетов среднего по агенству
     */
    public static final LocalDate AGENCY_STAT_START_DATE = LocalDate.of(2018, 1, 1);

    /**
     * Отступ в прошлое от текущей даты.
     * Штатно данные кубов есть за "позавчера", но отступаем с запасом
     * <p>
     * Используется, как fallback для вычисления максимально-допустимой даты
     */
    private static final int MAX_REPORT_DATE_GAP = 3;


    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public AgencyOfflineReportParametersService(PpcPropertiesSupport ppcPropertiesSupport) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    /**
     * Получить максимально-допустимую дату для отчетов.
     * Возвращает дату из ppc_properties (вычисленную по наличию таблиц с исходных данными)
     * или рассчитанную по {@link #MAX_REPORT_DATE_GAP}.
     *
     * @return граничное значение даты
     */
    public LocalDate getMaximumAvailableDate() {
        return ppcPropertiesSupport.find(MAX_REPORT_DATE_PROP_NAME)
                .map(AgencyOfflineReportParametersService::parseDate)
                .orElseGet(AgencyOfflineReportParametersService::getFallbackDate);
    }

    /**
     * Получить строковое представтление максимально-допустимой даты для отчетов в формате {@link #MAX_DATE_FORMATTER}
     *
     * @return строка с граничным значением даты
     * @see #getMaximumAvailableDate()
     */
    public String getMaximumAvailableDateAsString() {
        return getMaximumAvailableDate().format(MAX_DATE_FORMATTER);
    }

    /**
     * Сохранить в ppc_properties максимально-допустимую дату для отчетов
     *
     * @param date дата для сохранения
     */
    public void setMaximumAvailableDate(LocalDate date) {
        ppcPropertiesSupport.set(MAX_REPORT_DATE_PROP_NAME, date.format(MAX_DATE_FORMATTER));
    }

    private static LocalDate getFallbackDate() {
        return LocalDate.now().minusDays(MAX_REPORT_DATE_GAP);
    }

    @Nullable
    private static LocalDate parseDate(String value) {
        try {
            return LocalDate.parse(value, MAX_DATE_FORMATTER);
        } catch (DateTimeParseException e) {
            logger.error("Failed to parse datetime", e);
            return null;
        }
    }
}
