package ru.yandex.direct.core.entity.agencyofflinereport.service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.agencyofflinereport.container.AgencyOfflineReportArgs;
import ru.yandex.direct.core.entity.agencyofflinereport.container.AgencyOfflineReportJobParams;
import ru.yandex.direct.core.entity.agencyofflinereport.model.AgencyOfflineReport;
import ru.yandex.direct.core.entity.agencyofflinereport.model.AgencyOfflineReportKind;
import ru.yandex.direct.core.entity.agencyofflinereport.model.AgencyOfflineReportState;
import ru.yandex.direct.core.entity.agencyofflinereport.repository.AgencyOfflineReportRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.AGENCY_DASHBOARD_REPORT;

@Service
@ParametersAreNonnullByDefault
public class AgencyOfflineReportService {
    private final AgencyOfflineReportRepository agencyOfflineReportRepository;
    private final DbQueueRepository dbQueueRepository;
    private final RbacService rbacService;
    private final ShardHelper shardHelper;
    private final ClientService clientService;
    private final UserService userService;

    @Autowired
    public AgencyOfflineReportService(ShardHelper shardHelper,
                                      AgencyOfflineReportRepository agencyOfflineReportRepository,
                                      DbQueueRepository dbQueueRepository, RbacService rbacService,
                                      ClientService clientService,
                                      UserService userService) {
        this.shardHelper = shardHelper;
        this.agencyOfflineReportRepository = agencyOfflineReportRepository;
        this.dbQueueRepository = dbQueueRepository;
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.userService = userService;
    }

    /**
     * Заказать отчет.
     * Добавляет отчет в список, ставит задачу на построение.
     *
     * @param operator заказывающий отчет
     * @param agency   представитель агентства, на которое заказывают отчет
     * @param dateFrom дата с, включительно
     * @param dateTo   дата по, включительно
     */
    public void enqueueReportRequest(User operator, User agency, LocalDate dateFrom, LocalDate dateTo) {
        int operatorShard = clientService.getShardByClientIdStrictly(operator.getClientId());
        int agencyShard = clientService.getShardByClientIdStrictly(agency.getClientId());

        Long reportId = shardHelper.generateAgencyOfflineReportIds(1).get(0);

        AgencyOfflineReportArgs reportArgs = new AgencyOfflineReportArgs()
                .withAgencyLogin(agency.getLogin())
                .withReportKind(getReportKind(agency.getRepType()))
                .withDateFrom(dateFrom)
                .withDateTo(dateTo);

        Long reportOwnerUid = chooseReportOwnerUid(operator, agency);
        AgencyOfflineReport report = new AgencyOfflineReport()
                .withReportId(reportId)
                .withAgencyClientId(agency.getClientId())
                .withUid(reportOwnerUid)
                .withScheduledAt(LocalDateTime.now())
                .withReportState(AgencyOfflineReportState.NEW)
                .withArgs(reportArgs);

        AgencyOfflineReportJobParams jobParams = new AgencyOfflineReportJobParams(report)
                .withSubclients(getSubclients(agency.getUid()));

        Long jobId = dbQueueRepository
                .insertJob(operatorShard, AGENCY_DASHBOARD_REPORT, operator.getClientId(), reportOwnerUid, jobParams)
                .getId();
        reportArgs.withJobId(jobId);

        agencyOfflineReportRepository.addAgencyOfflineReport(agencyShard, report);
    }

    /**
     * Получить список отчетов.
     * Для менеджеров и агентств - возвращаются только те отчеты, которые заказал этот же оператор.
     *
     * @param operator оператор
     * @param agency   представитель агентства по которому хотят получить отчеты
     * @return список отчетов
     */
    public List<AgencyOfflineReport> getAgencyOfflineReports(User operator, User agency) {
        int shard = clientService.getShardByClientIdStrictly(agency.getClientId());

        Long reportOwnerUidFilter = operator.getRole().anyOf(RbacRole.AGENCY, RbacRole.MANAGER)
                ? chooseReportOwnerUid(operator, agency)
                : null;

        return agencyOfflineReportRepository.getAgencyOfflineReports(shard, agency.getClientId(), reportOwnerUidFilter);
    }

    /**
     * Получить отчет по его id.
     * NB! Не выполняет фильтрацию по uid'у владельца отчета
     *
     * @param agencyClientId ClientId агентства по которому построен отчет
     * @param reportId       идентификатор отчета
     * @return данные по отчету или {@code null} если отчет не найден
     */
    public AgencyOfflineReport getAgencyOfflineReport(ClientId agencyClientId, long reportId) {
        int shard = clientService.getShardByClientIdStrictly(agencyClientId);
        return agencyOfflineReportRepository.getAgencyOfflineReport(shard, reportId);
    }

    /**
     * Пометить обрабатывающийся отчет успешно построенным, сохранить ссылку на результат
     *
     * @param agencyClientId id агентства для которого строится отчет, нужно для определения шарда
     * @param reportId       id отчета
     * @param reportUrl      ссылка на готовый отчет
     */
    public void markProcessingReportAsReady(ClientId agencyClientId, long reportId, String reportUrl) {
        int shard = clientService.getShardByClientIdStrictly(agencyClientId);
        agencyOfflineReportRepository.markProcessingReportAsReady(shard, reportId, reportUrl);
    }

    /**
     * Пометить новый отчет как обрабатывающийся
     *
     * @param agencyClientId id агентства для которого строится отчет, нужно для определения шарда
     * @param reportId       id отчета
     */
    public void markNewReportAsProcessing(ClientId agencyClientId, long reportId) {
        int shard = clientService.getShardByClientIdStrictly(agencyClientId);
        agencyOfflineReportRepository.markNewReportAsProcessing(shard, reportId);
    }

    /**
     * Пометить отчет как ошибочный
     *
     * @param agencyClientId id агентства для которого строится отчет, нужно для определения шарда
     * @param reportId       id отчета
     */
    public void markReportFailed(ClientId agencyClientId, long reportId) {
        int shard = clientService.getShardByClientIdStrictly(agencyClientId);
        agencyOfflineReportRepository.markReportFailed(shard, reportId);
    }

    List<ClientId> getSubclients(Long agencyUid) {
        List<Long> subclientUids = rbacService.getAgencySubclients(agencyUid);
        return userService.massGetUser(subclientUids)
                .stream()
                .map(User::getClientId)
                .distinct()
                .sorted()
                .collect(Collectors.toList());
    }

    private static AgencyOfflineReportKind getReportKind(RbacRepType agencyRepType) {
        return agencyRepType == RbacRepType.CHIEF || agencyRepType == RbacRepType.MAIN
                ? AgencyOfflineReportKind.FULL
                : AgencyOfflineReportKind.PARTIAL;
    }

    static Long chooseReportOwnerUid(User operator, User agency) {
        // если агентство работает со своим представителем - берем его (представителя) uid
        return operator.getRole() == RbacRole.AGENCY && operator.getClientId().equals(agency.getClientId())
                ? agency.getUid()
                : operator.getUid();
    }
}
