package ru.yandex.direct.core.entity.apifinancetokens.repository;

import java.time.LocalDateTime;
import java.util.Optional;

import org.apache.commons.lang3.RandomStringUtils;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.enums.UsersApiOptionsApiAllowFinanceOperations;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.jooq.impl.DSL.defaultValue;
import static ru.yandex.direct.dbschema.ppc.tables.UsersApiOptions.USERS_API_OPTIONS;
import static ru.yandex.direct.dbschema.ppcdict.tables.ApiFinanceTokens.API_FINANCE_TOKENS;

@Repository
public class ApiFinanceTokensRepository {
    private final DslContextProvider ppcDslContextProvider;
    private final ShardHelper shardHelper;

    public ApiFinanceTokensRepository(DslContextProvider ppcDslContextProvider, ShardHelper shardHelper) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardHelper = shardHelper;
    }

    /**
     * Возвращает мастер-токен для указанного UID. Если токена нет либо он пустой - возвращает {@link Optional#empty()}
     */
    public Optional<String> getMasterToken(long uid) {
        return ppcDslContextProvider.ppcdict()
                .select(API_FINANCE_TOKENS.MASTER_TOKEN)
                .from(API_FINANCE_TOKENS)
                .where(API_FINANCE_TOKENS.UID.eq(uid))
                .fetchOptional(API_FINANCE_TOKENS.MASTER_TOKEN)
                .map(s -> isBlank(s) ? null : s);
    }

    /**
     * Создает и возвращает мастер-токен для указанного UID. Если токен уже есть - затирает старый токен новым.
     */
    public String createAndGetMasterToken(long uid) {
        checkState(checkFinanceOperationsAllowed(uid), "Financial operations have to be allowed for user");
        String token = RandomStringUtils.randomAlphanumeric(16);
        ppcDslContextProvider.ppcdict()
                .insertInto(API_FINANCE_TOKENS)
                .set(API_FINANCE_TOKENS.UID, uid)
                .set(API_FINANCE_TOKENS.MASTER_TOKEN, token)
                .set(API_FINANCE_TOKENS.MASTER_TOKEN_TIMECREATED, LocalDateTime.now())
                .onDuplicateKeyUpdate()
                .set(API_FINANCE_TOKENS.MASTER_TOKEN, MySQLDSL.values(API_FINANCE_TOKENS.MASTER_TOKEN))
                .set(API_FINANCE_TOKENS.MASTER_TOKEN_TIMECREATED,
                        MySQLDSL.values(API_FINANCE_TOKENS.MASTER_TOKEN_TIMECREATED))
                .execute();

        return token;
    }

    /**
     * Проверяем, что клиенту разрешены финансовые операции в апи
     */
    private boolean checkFinanceOperationsAllowed(Long uid) {
        int shard = shardHelper.getShardByClientUid(uid);
        return ppcDslContextProvider.ppc(shard)
                .selectOne()
                .from(USERS_API_OPTIONS)
                .where(USERS_API_OPTIONS.UID.eq(uid).and(USERS_API_OPTIONS.API_ALLOW_FINANCE_OPERATIONS
                        .eq(UsersApiOptionsApiAllowFinanceOperations.Yes)))
                .execute() != 0;
    }

    /**
     * Удаляет мастер-токен для указанного uid.
     */
    public void dropMasterToken(long uid) {
        ppcDslContextProvider.ppcdict()
                .insertInto(API_FINANCE_TOKENS)
                .set(API_FINANCE_TOKENS.UID, uid)
                .set(API_FINANCE_TOKENS.MASTER_TOKEN, "")
                .set(API_FINANCE_TOKENS.MASTER_TOKEN_TIMECREATED,
                        defaultValue(API_FINANCE_TOKENS.MASTER_TOKEN_TIMECREATED))
                .onDuplicateKeyUpdate()
                .set(API_FINANCE_TOKENS.MASTER_TOKEN,
                        MySQLDSL.values(API_FINANCE_TOKENS.MASTER_TOKEN))
                .set(API_FINANCE_TOKENS.MASTER_TOKEN_TIMECREATED,
                        MySQLDSL.values(API_FINANCE_TOKENS.MASTER_TOKEN_TIMECREATED))
                .execute();
    }
}
