package ru.yandex.direct.core.entity.auction.container;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.currency.Currency;

import static java.util.Objects.requireNonNull;

public class AdGroupForAuction {

    private final Campaign campaign;
    private final AdGroup adGroup;
    private final BannerWithSystemFields mainBanner;
    private final Vcard mainBannerVcard;
    private final List<Keyword> keywords;
    private final Currency currency;
    private final int bannerQuantity;
    private final Domain publisherDomain;
    private final String storeAppId;

    public static Builder builderFrom(AdGroupForAuction source) {
        return builder()
                .campaign(source.campaign)
                .adGroup(source.adGroup)
                .mainBanner(source.mainBanner)
                .mainBannerVcard(source.mainBannerVcard)
                .keywords(source.keywords)
                .currency(source.currency)
                .bannerQuantity(source.bannerQuantity)
                .publisherDomain(source.publisherDomain)
                .storeAppId(source.storeAppId);
    }

    public static Builder builder() {
        return new Builder();
    }

    @SuppressWarnings("squid:S00107") // конструктор приватный, делаем что хотим
    private AdGroupForAuction(Campaign campaign, AdGroup adGroup,
                              @Nullable BannerWithSystemFields mainBanner,
                              @Nullable Vcard mainBannerVcard,
                              @Nullable Domain publisherDomain,
                              @Nullable String storeAppId,
                              List<Keyword> keywords,
                              Currency currency, int bannerQuantity) {
        this.campaign = campaign;
        this.adGroup = adGroup;
        this.mainBanner = mainBanner;
        this.mainBannerVcard = mainBannerVcard;
        this.keywords = keywords;
        this.currency = currency;
        this.bannerQuantity = bannerQuantity;
        this.publisherDomain = publisherDomain;
        this.storeAppId = storeAppId;
    }

    public AdGroup getAdGroup() {
        return adGroup;
    }

    public Campaign getCampaign() {
        return campaign;
    }

    @Nullable
    public BannerWithSystemFields getBanner() {
        return mainBanner;
    }

    @Nullable
    public Vcard getVcard() {
        return mainBannerVcard;
    }

    public int getBannerQuantity() {
        return bannerQuantity;
    }

    @Nullable
    public Domain getPublisherDomain() {
        return publisherDomain;
    }

    public List<Keyword> getKeywords() {
        return keywords;
    }

    public Currency getCurrency() {
        return currency;
    }

    /**
     * Значение присутствует только для групп РМП, для других – {@code null}.
     */
    @Nullable
    public String getStoreAppId() {
        return storeAppId;
    }

    /**
     * Определяет, соответствует ли группа формату баннера.
     * <p>
     * Критерии:
     * <ul>
     * <li>количество баннеров в группе = 1</li>
     * <li>стратегия != 'different_places'</li>
     * <li>у картинки баннера установлен БК ID</li>
     * </ul>
     */
    public boolean isBannerFormat() {
        boolean bannerFormat = false;
        boolean onlyOneBannerInAdGroup = getBannerQuantity() == 1;
        BannerWithSystemFields banner = getBanner();
        if (banner != null && banner instanceof TextBanner) {
            // повторена логика из Perl
            Long bsImageId = Optional.ofNullable(((TextBanner) banner).getImageBsBannerId())
                    .orElse(0L);
            boolean bannerImageIsKnownToBs = bsImageId > 0L;
            boolean networkBidsDependsOnSearch =
                    !getCampaign().getStrategy().isDifferentPlaces();
            bannerFormat = onlyOneBannerInAdGroup && bannerImageIsKnownToBs && networkBidsDependsOnSearch;
        }
        return bannerFormat;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("AdGroupForAuction{");
        sb.append("campaign=").append(campaign);
        sb.append(", adGroup=").append(adGroup);
        sb.append(", mainBanner=").append(mainBanner);
        sb.append(", mainBannerVcard=").append(mainBannerVcard);
        sb.append(", keywords=").append(keywords);
        sb.append(", currency=").append(currency);
        sb.append(", bannerQuantity=").append(bannerQuantity);
        sb.append(", publisherDomain=").append(publisherDomain);
        sb.append(", storeAppId='").append(storeAppId).append('\'');
        sb.append(", banner=").append(getBanner());
        sb.append(", vcard=").append(getVcard());
        sb.append(", bannerFormat=").append(isBannerFormat());
        sb.append('}');
        return sb.toString();
    }

    public static final class Builder {
        private Campaign campaign;
        private AdGroup adGroup;
        private BannerWithSystemFields mainBanner;
        private Vcard mainBannerVcard;
        private List<Keyword> keywords;
        private Currency currency;
        private Integer bannerQuantity;
        private Domain publisherDomain;
        private String storeAppId;

        Builder() {
        }

        public Builder campaign(Campaign campaign) {
            this.campaign = campaign;
            return this;
        }

        public Builder adGroup(AdGroup adGroup) {
            this.adGroup = adGroup;
            return this;
        }

        public Builder mainBanner(BannerWithSystemFields mainBanner) {
            this.mainBanner = mainBanner;
            return this;
        }

        public Builder mainBannerVcard(Vcard mainBannerVcard) {
            this.mainBannerVcard = mainBannerVcard;
            return this;
        }

        public Builder keywords(List<Keyword> keywords) {
            this.keywords = keywords;
            return this;
        }

        public Builder currency(Currency currency) {
            this.currency = currency;
            return this;
        }

        public Builder bannerQuantity(int bannerQuantity) {
            this.bannerQuantity = bannerQuantity;
            return this;
        }

        public Builder publisherDomain(Domain publisherDomain) {
            this.publisherDomain = publisherDomain;
            return this;
        }

        /**
         * Идентификатор приложения для группы РМП. Для групп других типов должно быть {@code null}.
         */
        public Builder storeAppId(String bundleId) {
            this.storeAppId = bundleId;
            return this;
        }

        public AdGroupForAuction build() {
            return new AdGroupForAuction(
                    requireNonNull(campaign),
                    requireNonNull(adGroup),
                    mainBanner,
                    mainBannerVcard,
                    publisherDomain,
                    storeAppId,
                    requireNonNull(keywords),
                    requireNonNull(currency),
                    requireNonNull(bannerQuantity));
        }
    }
}
