package ru.yandex.direct.core.entity.auction.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bsauction.BasicBsRequestPhrase;
import ru.yandex.direct.bsauction.BidCalculationMethod;
import ru.yandex.direct.bsauction.BsCpcPrice;
import ru.yandex.direct.bsauction.BsRequest;
import ru.yandex.direct.bsauction.BsRequestPhraseStat;
import ru.yandex.direct.bsauction.BsResponse;
import ru.yandex.direct.bsauction.BsTrafaretClient;
import ru.yandex.direct.bsauction.FullBsTrafaretResponsePhrase;
import ru.yandex.direct.bsauction.PositionalBsTrafaretResponsePhrase;
import ru.yandex.direct.bshistory.HistoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.auction.BsRequestPhraseWrapperAdditionalData;
import ru.yandex.direct.core.entity.auction.container.AdGroupForAuction;
import ru.yandex.direct.core.entity.auction.container.BsRequestPhraseWrapper;
import ru.yandex.direct.core.entity.auction.container.bs.Block;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.container.bs.Position;
import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.auction.exception.BsAuctionUnavailableException;
import ru.yandex.direct.core.entity.auction.type.BsAuctionRequestTypeSupportFacade;
import ru.yandex.direct.core.entity.auction.utils.BsAuctionUtils;
import ru.yandex.direct.core.entity.autobroker.model.AutoBrokerResult;
import ru.yandex.direct.core.entity.autobroker.service.AutoBrokerCalculator;
import ru.yandex.direct.core.entity.autobroker.service.AutoBrokerCalculatorProviderService;
import ru.yandex.direct.core.entity.banner.model.BannerWithBody;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTitle;
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.type.href.BannerDomainRepository;
import ru.yandex.direct.core.entity.bids.container.interpolator.CapKey;
import ru.yandex.direct.core.entity.bids.interpolator.InterpolatorService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.keyword.model.ForecastCtr;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordForecastService;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.auction.utils.BsAuctionUtils.keywordIsKnownToBs;
import static ru.yandex.direct.core.entity.auction.utils.BsAuctionUtils.prepareText;
import static ru.yandex.direct.core.entity.auction.utils.BsAuctionUtils.replaceTemplateText;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getValueIfAssignable;
import static ru.yandex.direct.core.entity.bids.interpolator.InterpolatorService.HALF_OF_THE_DEFAULT_MAX_TRAFFIC_VOLUME;
import static ru.yandex.direct.integrations.configuration.IntegrationsConfiguration.BS_TRAFARET_AUCTION_CLIENT;
import static ru.yandex.direct.integrations.configuration.IntegrationsConfiguration.BS_TRAFARET_AUCTION_CLIENT_WEB;

@ParametersAreNonnullByDefault
@Service
public class BsAuctionService {
    private static final Logger logger = LoggerFactory.getLogger(BsAuctionService.class);

    // значения магических констант пришли из Perl.
    // Описание default'ов есть в https://st.yandex-team.ru/DIRECT-46905
    public static final double DEFAULT_PREMIUM_CTR = 0.2;
    public static final double DEFAULT_GUARANTEE_CTR = 0.03;

    /**
     * при формировании пачки фраз для запроса в торги, <кол-во фраз>*<кол-во баннеров в группе> не должно превышать 150
     */
    private static final int BANNER_MUL_PHRASES_CHUNK_LIMIT = 150;
    /**
     * максимальное количество фраз в запросе в торги
     */
    public static final int PHRASES_CHUNK_LIMIT = 10;

    private final KeywordForecastService keywordForecastService;
    private final AutoBrokerCalculatorProviderService autoBrokerCalculatorProviderService;
    private final BsTrafaretClient bsTrafaretClient;
    private final BsTrafaretClient bsTrafaretClientWeb;
    private final boolean useWebClientConfig;
    private final CampaignService campaignService;
    private final InterpolatorService interpolatorService;
    private final BsAuctionRequestTypeSupportFacade bsAuctionRequestTypeSupportFacade;
    private final BannerDomainRepository newBannerDomainRepository;

    @Autowired
    public BsAuctionService(KeywordForecastService keywordForecastService,
                            AutoBrokerCalculatorProviderService autoBrokerCalculatorProviderService,
                            @Qualifier(BS_TRAFARET_AUCTION_CLIENT) BsTrafaretClient bsTrafaretClient,
                            @Qualifier(BS_TRAFARET_AUCTION_CLIENT_WEB) BsTrafaretClient bsTrafaretClientWeb,
                            @Value("${bsauction.use-web-client:false}") boolean useWebClientConfig,
                            CampaignService campaignService,
                            InterpolatorService interpolatorService,
                            BsAuctionRequestTypeSupportFacade bsAuctionRequestTypeSupportFacade,
                            BannerDomainRepository newBannerDomainRepository) {
        this.keywordForecastService = keywordForecastService;
        this.autoBrokerCalculatorProviderService = autoBrokerCalculatorProviderService;
        this.bsTrafaretClient = bsTrafaretClient;
        this.bsTrafaretClientWeb = bsTrafaretClientWeb;
        this.campaignService = campaignService;
        this.interpolatorService = interpolatorService;
        this.bsAuctionRequestTypeSupportFacade = bsAuctionRequestTypeSupportFacade;
        this.newBannerDomainRepository = newBannerDomainRepository;
        this.useWebClientConfig = useWebClientConfig;
    }

    /**
     * Для фраз из переданных групп запрашиваем данные из Торгов БК.
     * Для групп с взведённым флагом "Мало показов" в Торги не ходим.
     * Если у группы объявлений нет баннера, то тоже не ходим в Торги.
     *
     * @throws BsAuctionUnavailableException если не по всем запрашенным фразам удалось получить ответ от Торгов
     * @see BsTrafaretClient
     */
    public List<KeywordBidBsAuctionData> getBsResults(ClientId clientId,
                                                      List<AdGroupForAuction> keywords)
            throws BsAuctionUnavailableException {
        List<BsRequest<BsRequestPhraseWrapper>> bsRequests = getBsRequests(clientId, keywords);
        if (bsRequests.isEmpty()) {
            return emptyList();
        }

        IdentityHashMap<? extends BsRequest<BsRequestPhraseWrapper>, BsResponse<BsRequestPhraseWrapper,
                PositionalBsTrafaretResponsePhrase>>
                auctionResults = getBsTrafaretClient().getAuctionResults(bsRequests);
        checkThatAllResponsesAreSuccessful(auctionResults.values());

        List<Campaign> campaigns = StreamEx.of(keywords).map(AdGroupForAuction::getCampaign).distinct().toList();
        AutoBrokerCalculator autoBrokerCalculator =
                autoBrokerCalculatorProviderService.getAutoBrokerCalculatorForCampaigns(campaigns);
        Map<Long, WalletRestMoney> walletBalancesByCampaignId =
                campaignService.getWalletsRestMoney(clientId, campaigns);

        return StreamEx.ofValues(auctionResults)
                .map(BsResponse::getSuccessResults)
                .flatCollection(Map::entrySet)
                .mapToEntry(Map.Entry::getKey, Map.Entry::getValue)
                .mapKeyValue((wrapper, bsResponse) -> convertBsResponse(wrapper, bsResponse, autoBrokerCalculator,
                        walletBalancesByCampaignId))
                .toList();
    }

    /**
     * Делает запрос в БК и обрабатывает ответ с помощью
     * {@link InterpolatorService#getInterpolatedTrafaretBidItems(CapKey, List, List, CurrencyCode)}
     */
    public List<KeywordTrafaretData> getBsTrafaretResults(ClientId clientId, List<AdGroupForAuction> keywords)
            throws BsAuctionUnavailableException {
        return getBsTrafaretResults(getBsRequests(clientId, keywords));
    }

    public List<KeywordTrafaretData> getBsTrafaretResults(List<BsRequest<BsRequestPhraseWrapper>> bsRequests)
            throws BsAuctionUnavailableException {
        if (bsRequests.isEmpty()) {
            return emptyList();
        }

        IdentityHashMap<? extends BsRequest<BsRequestPhraseWrapper>,
                BsResponse<BsRequestPhraseWrapper, FullBsTrafaretResponsePhrase>>
                auctionResults = getBsTrafaretClient().getAuctionResultsWithPositionCtrCorrection(bsRequests);
        checkThatAllResponsesAreSuccessful(auctionResults.values());

        return StreamEx.ofValues(auctionResults)
                .map(BsResponse::getSuccessResults)
                .flatCollection(Map::entrySet)
                .mapToEntry(Map.Entry::getKey, Map.Entry::getValue)
                .mapKeyValue(this::convertBsTrafaretResponse)
                .toList();
    }

    private BsTrafaretClient getBsTrafaretClient() {
        return useWebClientConfig ?
                bsTrafaretClientWeb : bsTrafaretClient;
    }

    private List<BsRequest<BsRequestPhraseWrapper>> getBsRequests(ClientId clientId, List<AdGroupForAuction> keywords) {
        // для неизвестных в БК фраз попробуем найти информацию о среднем CTR
        IdentityHashMap<Keyword, ForecastCtr> forecastCtrs = getForecastCtrForUnknownKeywords(keywords);

        var bannerIds = keywords.stream()
                .map(keyword -> Optional.ofNullable(keyword.getBanner())
                        .map(BannerWithSystemFields::getId)
                        .orElse(null))
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        Map<Long, String> filterDomainByBid = newBannerDomainRepository.getFilterDomainByBannerIdMap(clientId,
                bannerIds);
        return StreamEx.of(keywords)
                .map(adGroupForAuction -> toBsRequest(adGroupForAuction, forecastCtrs, filterDomainByBid))
                .nonNull()
                .flatMap(Collection::stream)
                .toList();
    }

    private void checkThatAllResponsesAreSuccessful(Collection<? extends BsResponse> auctionResults)
            throws BsAuctionUnavailableException {
        List<? extends BsResponse> unsuccessfulResponses = StreamEx.of(auctionResults)
                .remove(BsResponse::isSuccessful)
                .toList();

        if (!unsuccessfulResponses.isEmpty()) {
            logger.error("Can't get BS auction response for all phrases. Got errors in next bs-auction responses: {}",
                    unsuccessfulResponses);
            throw new BsAuctionUnavailableException();
        }
    }

    /**
     * Для тех {@link Keyword}, которые не известны в Торгах ({@code phraseBsId == 0}),
     * достаём данные о прогнозируемом CTR.
     */
    private IdentityHashMap<Keyword, ForecastCtr> getForecastCtrForUnknownKeywords(List<AdGroupForAuction> keywords) {
        List<Keyword> keywordsUnknownToBs = StreamEx.of(keywords)
                .flatCollection(AdGroupForAuction::getKeywords)
                .remove(BsAuctionUtils::keywordIsKnownToBs)
                .toList();

        return keywordForecastService.getForecast(keywordsUnknownToBs);
    }

    /**
     * @return {@code null}, если для фразы не идём в Торги. Например, отсутствует Баннер.
     */
    @Nullable
    List<BsRequest<BsRequestPhraseWrapper>> toBsRequest(AdGroupForAuction adGroupForAuction,
                                                        IdentityHashMap<Keyword, ForecastCtr> forecastCtrsByKeyword,
                                                        Map<Long, String> filterDomainByBid) {
        Campaign campaign = checkNotNull(adGroupForAuction.getCampaign(), "Campaign not found");
        AdGroup adGroup = checkNotNull(adGroupForAuction.getAdGroup(), "AdGroup not found");

        @Nullable
        BannerWithSystemFields banner = adGroupForAuction.getBanner();
        Collection<Keyword> phrases = adGroupForAuction.getKeywords();

        if (banner == null || phrases == null) {
            // Не ходим в Торги за данными, если не известен баннер, или отсутствуют фразы (автотаргетинг).
            // Без данных баннера Торги отдают плохие данные. Без фраз ходить в торги пока тоже не имеет смысла
            return null;
        }

        if (banner instanceof ImageBanner || banner instanceof CpcVideoBanner) {
            // Для графических и кликовых видео объявлений не ходим в Торги
            return null;
        }

        logger.trace("Creating bsRequest for adGroup {}", adGroupForAuction);
        int partitionSize = calcPartitionSize(adGroupForAuction.getBannerQuantity());

        List<List<Keyword>> partitions = Lists.partition(new ArrayList<>(phrases), partitionSize);
        List<BsRequest<BsRequestPhraseWrapper>> result = new ArrayList<>(partitions.size());
        for (List<Keyword> partition : partitions) {

            BsRequest<BsRequestPhraseWrapper> request = new BsRequest<>();
            request
                    .withOrderId(campaign.getOrderId())
                    .withNoExtendedGeotargeting(campaign.getOpts().contains(CampaignOpts.NO_EXTENDED_GEOTARGETING))
                    .withTimetable(Boolean.TRUE.equals(campaign.getFairAuction()))
                    .withOnlyMobilePages(adGroup.getType() == AdGroupType.MOBILE_CONTENT);
            if (adGroup.getGeo() != null) {
                request.withRegionIds(adGroup.getGeo());
            }

            // заполняем числовой ISO код валюты, если валюта не YND_FIXED
            request.withCurrency(adGroupForAuction.getCurrency());

            // баннер может быть без домена, только с визиткой
            String domain = filterDomainByBid.getOrDefault(banner.getId(),
                    getValueIfAssignable(banner, BannerWithHref.DOMAIN));
            if (domain == null && adGroupForAuction.getVcard() != null) {
                domain = phoneToDomain(adGroupForAuction.getVcard().getPhone());
            }
            // для РМП групп требуется указывать домен издателя или
            // – если последний отсутствует – идентификатор приложения
            if (adGroup.getType() == AdGroupType.MOBILE_CONTENT) {
                domain = Optional.ofNullable(adGroupForAuction.getPublisherDomain())
                        .map(Domain::getDomain)
                        .orElse(Optional.ofNullable(adGroupForAuction.getStoreAppId())
                                .orElse(domain));
                logger.debug("For Mobile_content adGroup {} use domain {}. Publisher domain: {}. Store app ID: {}",
                        adGroup.getId(), domain, adGroupForAuction.getPublisherDomain(),
                        adGroupForAuction.getStoreAppId());
            }
            request.withDomain(domain);

            //todo maxlog: для графических баннеров не ходим в торги, ветка кода else мёртвая
            if (banner instanceof TextBanner || banner instanceof MobileAppBanner
                    || banner instanceof ContentPromotionBanner) {
                // если баннер не графический, то заполняем ban-head и ban-body
                String bannerHead = replaceTemplateText(((BannerWithTitle) banner).getTitle());
                String bannerBody = replaceTemplateText(((BannerWithBody) banner).getBody());

                logger.debug("Banner texts for adGroup(id={}): head=\"{}\", body=\"{}\"",
                        adGroup.getId(), bannerHead, bannerBody);
                request.withBannerHead(bannerHead)
                        .withBannerBody(bannerBody);
            } else {
                // иначе ban-body не заполняем, а в ban-head кладём первую (в лексикографическом смысле) фразу
                // тут ищем по всем фразам группы, а не только по находящимся в текущей партиции
                String firstPhrase = StreamEx.of(phrases)
                        .map(Keyword::getPhrase)
                        .sorted()
                        .findFirst()
                        .orElseThrow(() -> new IllegalArgumentException(
                                "Keywords list can't be empty or contains null phrases"));
                String bannerHead = prepareText(firstPhrase);
                logger.debug("TextBanner is image_ad. Use {} as ban-head", bannerHead);
                request.withBannerHead(bannerHead);
            }

            // повторена логика из Perl
            // Про параметр operation можно чуть больше узнать в описании взаимодействия Директа с Торгами:
            // https://wiki.yandex-team.ru/Direkt/Auction/
            if (!adGroupForAuction.isBannerFormat()) {
                // подробнее про operation=3 в "YABS-48814: Ещё один вид торгов для групп - псевдочестные, operation=3"
                logger.debug("Use GROUP_BID for adGroup(id={}, type={})", adGroup.getId(), adGroup.getType());
                request.withBidCalcMethod(BidCalculationMethod.GROUP_BID);
            }

            bsAuctionRequestTypeSupportFacade.setAdditionalQueryParams(request, adGroup, campaign);

            List<BsRequestPhraseWrapper> bsPhraseRequests = StreamEx.of(partition)
                    .map(keyword -> convertToBsPhrase(keyword, forecastCtrsByKeyword.get(keyword),
                            adGroupForAuction))
                    .toList();
            request.withPhrases(bsPhraseRequests);
            result.add(request);
        }

        return result;
    }

    /**
     * Преобразует телефон в нашей нотации в домен
     */
    private String phoneToDomain(Phone phone) {
        String fullPhone = phone.getCountryCode() + phone.getCityCode() + phone.getPhoneNumber();
        String result = fullPhone.replaceAll("[^\\d]", "");
        return result + ".phone";
    }


    private int calcPartitionSize(int bannerQuantity) {
        if (0 < bannerQuantity && bannerQuantity <= BANNER_MUL_PHRASES_CHUNK_LIMIT) {
            return Math.min(BANNER_MUL_PHRASES_CHUNK_LIMIT / bannerQuantity, PHRASES_CHUNK_LIMIT);
        } else {
            return PHRASES_CHUNK_LIMIT;
        }
    }

    private BsRequestPhraseWrapper convertToBsPhrase(Keyword keyword, @Nullable ForecastCtr forecastCtr,
                                                     AdGroupForAuction adGroupForAuction) {
        BsRequestPhraseStat bsRequestPhraseStat;
        if (keywordIsKnownToBs(keyword)) {
            String phraseIdHistoryPrepended;
            if (adGroupForAuction.isBannerFormat()) {
                String bannerHistory = Optional.ofNullable(keyword.getPhraseIdHistory())
                        .map(HistoryUtils::convertHistoryForBanner)
                        .orElse(null);

                phraseIdHistoryPrepended = HistoryUtils
                        .prependHistoryForBanner(bannerHistory, adGroupForAuction.getBanner().getBsBannerId(),
                                keyword.getPhraseBsId());
            } else {
                Long adGroupId = adGroupForAuction.getAdGroup().getId();
                String groupHistory = Optional.ofNullable(keyword.getPhraseIdHistory())
                        .map(phraseIdHistory -> HistoryUtils.convertHistoryForAdGroup(phraseIdHistory, adGroupId))
                        .orElse(null);

                phraseIdHistoryPrepended =
                        HistoryUtils.prependHistoryForAdGroup(groupHistory, adGroupId, keyword.getPhraseBsId());
            }
            bsRequestPhraseStat = BsRequestPhraseStat.getByHistory(phraseIdHistoryPrepended);
        } else {
            bsRequestPhraseStat = getRequestPhraseStatForUnknownToBs(forecastCtr, keyword.getShowsForecast());
        }
        return new BsRequestPhraseWrapper(
                new BasicBsRequestPhrase()
                        .withText(prepareText(keyword.getPhrase()))
                        .withStat(bsRequestPhraseStat))
                .withAdditionalData(new BsRequestPhraseWrapperAdditionalData(adGroupForAuction, keyword));
    }

    public BsRequestPhraseStat getRequestPhraseStatForUnknownToBs(@Nullable ForecastCtr forecastCtr,
                                                                  @Nullable Long keywordShowsForecast) {
        Optional<ForecastCtr> forecastCtrOptional = Optional.ofNullable(forecastCtr);
        double guaranteeCtr = forecastCtrOptional.map(ForecastCtr::getGuaranteeCtr)
                .filter(ctr -> ctr > 0)
                .orElse(DEFAULT_GUARANTEE_CTR);
        double premiumCtr = forecastCtrOptional.map(ForecastCtr::getPremiumCtr)
                .filter(ctr -> ctr > 0)
                .orElse(DEFAULT_PREMIUM_CTR);

        long showsForecast = Optional.ofNullable(keywordShowsForecast).orElse(0L);
        long showsForecastGuarantee = guaranteeCtr < 0.005 ? 300 : showsForecast;
        long clicksForecastGuarantee = Math.round(showsForecastGuarantee * guaranteeCtr);
        long showsForecastPremium = guaranteeCtr < 0.005 ? 300 : showsForecast;
        long clicksForecastPremium = Math.round(showsForecastPremium * premiumCtr);

        return BsRequestPhraseStat.getByForecast(showsForecastGuarantee, clicksForecastGuarantee, showsForecastPremium,
                clicksForecastPremium);
    }

    private Position convertCpcPriceToPosition(BsCpcPrice bsCpcPrice) {
        return new Position(bsCpcPrice.getCpc(), bsCpcPrice.getPrice());
    }

    KeywordBidBsAuctionData convertBsResponse(BsRequestPhraseWrapper wrapper,
                                              PositionalBsTrafaretResponsePhrase bsResponsePhrase,
                                              AutoBrokerCalculator autoBrokerCalculator,
                                              Map<Long, WalletRestMoney> walletBalancesByCampaignIds) {

        Money brokerPrice = null;
        AdGroupForAuction adGroupForAuction = wrapper.getAdGroupForAuction();
        if (!adGroupForAuction.getAdGroup().getBsRarelyLoaded()) {
            // цена на поиске расчитывается только если у группы не взведён флаг "Мало показов"
            Campaign campaign = adGroupForAuction.getCampaign();
            Keyword keyword = wrapper.getKeyword();
            WalletRestMoney wallet = walletBalancesByCampaignIds.get(campaign.getId());
            AutoBrokerResult autoBrokerResult = autoBrokerCalculator
                    .calculatePrices(campaign, keyword, bsResponsePhrase, wallet);
            if (autoBrokerResult.getBrokerPrice().greaterThanZero()) {
                // проставляем цену на поиске, если от АвтоБрокера она пришла ненулевой
                brokerPrice = autoBrokerResult.getBrokerPrice();
            }
        }

        Money minPrice = bsResponsePhrase.getMinPrice();
        return new KeywordBidBsAuctionData()
                .withKeyword(wrapper.getKeyword())
                .withContextStopFlag(false)
                .withMinPrice(minPrice)
                .withBroker(brokerPrice)
                .withPremium(new Block(StreamEx.of(bsResponsePhrase.getPremium())
                        .map(this::convertCpcPriceToPosition)
                        .sorted()
                        .toList()))
                .withGuarantee(new Block(StreamEx.of(bsResponsePhrase.getGuarantee())
                        .map(this::convertCpcPriceToPosition)
                        .sorted()
                        .toList()));
    }

    private KeywordTrafaretData convertBsTrafaretResponse(BsRequestPhraseWrapper wrapper,
                                                          FullBsTrafaretResponsePhrase bsResponsePhrase) {
        List<TrafaretBidItem> bidItems = StreamEx.of(bsResponsePhrase.getBidItems())
                .map(t -> new TrafaretBidItem()
                        .withPositionCtrCorrection(t.getPositionCtrCorrection())
                        .withBid(t.getBid())
                        .withPrice(t.getPrice()))
                .sorted(Comparator.comparing(TrafaretBidItem::getPositionCtrCorrection).reversed())
                .toList();
        List<TrafaretBidItem> interpolatedBidItems = interpolate(bidItems, wrapper);

        return new KeywordTrafaretData()
                .withKeyword(wrapper.getKeyword())
                .withPhrase(wrapper.getPhrase())
                .withBidItems(interpolatedBidItems);
    }

    /**
     * Сглаживает и интерполирует ответ от БК
     *
     * @param bidItems сырой ответ от БК
     * @param wrapper  запрос к БК
     * @see InterpolatorService
     */
    List<TrafaretBidItem> interpolate(List<TrafaretBidItem> bidItems, BsRequestPhraseWrapper wrapper) {
        String domain = wrapper.getBannerDomain();
        boolean isMcBanner = wrapper.isMcBanner();

        if (isMcBanner) {
            return List.of(bidItems.stream()
                    .filter(item -> item.getPositionCtrCorrection() == 1_000_000)
                    .findFirst()
                    // если по каким-то причинам БК нарушили контракт, и не прислали X == 1_000_000 для ГО на поиске
                    .orElseThrow(() -> new IllegalArgumentException(
                            "No bs_trafaret data for mcbanner")));
        }

        boolean isTop = StreamEx.of(bidItems)
                .map(TrafaretBidItem::getPositionCtrCorrection)
                .anyMatch(x -> x > HALF_OF_THE_DEFAULT_MAX_TRAFFIC_VOLUME);
        String phrase = wrapper.getPhrase();
        boolean isExactMatch = phrase.trim().startsWith("\"");
        CapKey capKey = new CapKey(isExactMatch, isTop ? 1 : 0, domain);

        List<TrafaretBidItem> interpolatedBidItems =
                interpolatorService.getInterpolatedTrafaretBidItems(capKey, bidItems, null,
                        wrapper.getCurrency().getCode());

        return StreamEx.of(interpolatedBidItems)
                .mapToEntry(TrafaretBidItem::getPositionCtrCorrection, Function.identity())
                //при совпадении PositionCtrCorrection выбираем тот, для которого меньше ставка
                .collapseKeys((t1, t2) -> t1.getBid().compareTo(t2.getBid()) > 0 ? t2 : t1)
                .values()
                .toList();
    }
}
