package ru.yandex.direct.core.entity.auction.utils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import one.util.streamex.StreamEx;

import ru.yandex.direct.bsauction.BsTrafaretClient;
import ru.yandex.direct.core.entity.auction.container.bs.Block;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.auction.container.bs.KeywordTrafaretData;
import ru.yandex.direct.core.entity.auction.container.bs.Position;
import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Правила преобразования разных форматов ответа Торгов друг в друга
 */
public class BsAuctionConverter {
    /**
     * Преобразовать ответ от трафарентных торгов в ответ позиционных торгов. Для преобразования используется {@link BsTrafaretClient#PLACE_TRAFARET_MAPPING}
     *
     * @param trafaretData ответ от трафаретных торгов
     * @param currency     валюта клиента
     * @return позиционный ответ от торгов. Отсутствуют значения: <li>{@link KeywordBidBsAuctionData#broker} <li>{@link KeywordBidBsAuctionData#minPrice} <li>{@link KeywordBidBsAuctionData#contextStopFlag}
     */
    public static List<KeywordBidBsAuctionData> convertToPositionsAuctionData(List<KeywordTrafaretData> trafaretData,
                                                                              Currency currency) {
        Money moreThanMaxBid = Money.unreachableBid(currency);
        return mapList(trafaretData, trafaretItem -> convertToPositionsAuctionData(trafaretItem, moreThanMaxBid));
    }

    /**
     * Преобразовать ответ от трафарентных торгов в ответ позиционных торгов.
     */
    public static KeywordBidBsAuctionData convertToPositionsAuctionData(KeywordTrafaretData keywordTrafaretData,
                                                                        Currency currency) {
        Money moreThanMaxBid = Money.unreachableBid(currency);
        return convertToPositionsAuctionData(keywordTrafaretData, moreThanMaxBid);
    }

    private static KeywordBidBsAuctionData convertToPositionsAuctionData(KeywordTrafaretData keywordTrafaretData,
                                                                         Money moreThanMaxBid) {
        KeywordBidBsAuctionData result = new KeywordBidBsAuctionData();

        List<Position> premiumPositions = new ArrayList<>();
        TrafaretBidItem p1 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.PREMIUM1),
                        keywordTrafaretData, moreThanMaxBid);
        TrafaretBidItem p2 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.PREMIUM2),
                        keywordTrafaretData, moreThanMaxBid);
        TrafaretBidItem p3 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.PREMIUM3),
                        keywordTrafaretData, moreThanMaxBid);
        TrafaretBidItem p4 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.PREMIUM4),
                        keywordTrafaretData, moreThanMaxBid);
        premiumPositions.add(new Position(p1.getPrice(), p1.getBid()));
        premiumPositions.add(new Position(p2.getPrice(), p2.getBid()));
        premiumPositions.add(new Position(p3.getPrice(), p3.getBid()));
        premiumPositions.add(new Position(p4.getPrice(), p4.getBid()));
        Block premium = new Block(premiumPositions);

        List<Position> guaranteePositions = new ArrayList<>();
        TrafaretBidItem g1 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.GUARANTEE1),
                        keywordTrafaretData, moreThanMaxBid);
        TrafaretBidItem g2 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.GUARANTEE2),
                        keywordTrafaretData, moreThanMaxBid);
        TrafaretBidItem g3 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.GUARANTEE3),
                        keywordTrafaretData, moreThanMaxBid);
        TrafaretBidItem g4 =
                getTrafaretBidItem(BsTrafaretClient.PLACE_TRAFARET_MAPPING.get(BsTrafaretClient.Place.GUARANTEE4),
                        keywordTrafaretData, moreThanMaxBid);
        guaranteePositions.add(new Position(g1.getPrice(), g1.getBid()));
        guaranteePositions.add(new Position(g2.getPrice(), g2.getBid()));
        guaranteePositions.add(new Position(g3.getPrice(), g3.getBid()));
        guaranteePositions.add(new Position(g4.getPrice(), g4.getBid()));
        Block guarantee = new Block(guaranteePositions);

        return result.withKeyword(keywordTrafaretData.getKeyword())
                .withPremium(premium)
                .withGuarantee(guarantee);
    }

    private static TrafaretBidItem getTrafaretBidItem(long positionCtrCorrection,
                                                      KeywordTrafaretData keywordTrafaretData,
                                                      Money moreThanMaxBid) {
        return StreamEx.of(keywordTrafaretData.getBidItems())
                .filter(t -> t.getPositionCtrCorrection() >= positionCtrCorrection)
                .min(Comparator.comparing(TrafaretBidItem::getBid))
                .orElse(new TrafaretBidItem()
                        .withBid(moreThanMaxBid)
                        .withPrice(moreThanMaxBid)
                        .withPositionCtrCorrection(positionCtrCorrection));
    }
}
