package ru.yandex.direct.core.entity.autobroker.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.bsauction.BsCpcPrice;
import ru.yandex.direct.bsauction.PositionalBsTrafaretResponsePhrase;
import ru.yandex.direct.core.entity.autobroker.model.AutoBrokerResult;
import ru.yandex.direct.core.entity.autobroker.model.BrokerPrice;
import ru.yandex.direct.core.entity.bids.service.BidBase;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.keyword.model.Place;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.MoneyUtils;
import ru.yandex.direct.libs.timetarget.ProductionCalendar;
import ru.yandex.direct.libs.timetarget.TimeTarget;

import static com.google.common.base.Preconditions.checkArgument;
import static java.math.BigDecimal.ZERO;
import static java.util.Arrays.asList;

/**
 * {@link AutoBrokerCalculator} при создании получает всю необходимую информацию для вычисления цен автоброкера
 * для ставок указанных кампаний. Это позволяет снизить накладные расходы на получение, например,
 * количества израсходованных средств на кампанию за текущий день.
 */
public class AutoBrokerCalculator {
    private static final Logger logger = LoggerFactory.getLogger(AutoBrokerCalculator.class);

    private final Set<Long> expectedCampaignIds;
    private final Map<Long, GeoTimezone> geoTimezoneByTimezoneId;
    private final Map<GeoTimezone, ProductionCalendar> productionCalendarByGeo;

    AutoBrokerCalculator(
            Set<Long> expectedCampaignIds, Map<Long, GeoTimezone> geoTimezoneByTimezoneId,
            Map<GeoTimezone, ProductionCalendar> productionCalendarByGeo) {
        this.expectedCampaignIds = expectedCampaignIds;
        this.geoTimezoneByTimezoneId = geoTimezoneByTimezoneId;
        this.productionCalendarByGeo = productionCalendarByGeo;
    }

    public AutoBrokerResult calculatePrices(Campaign campaign, BidBase bid, PositionalBsTrafaretResponsePhrase bsResult,
                                            WalletRestMoney walletRestMoney) {
        checkArgument(expectedCampaignIds.contains(campaign.getId()),
                "Unexpected campaignId %s. Expected one of following: %s", campaign.getId(), expectedCampaignIds);
        DbStrategy dbStrategy = campaign.getStrategy();
        AuctionStrategyMode auctionStrategyMode = getAuctionStrategyMode(dbStrategy);
        CurrencyCode currency = campaign.getCurrency();
        //при отстутствии цены в базе, она достается равным null
        Money keywordBid = bid.getPrice() != null ? Money.valueOf(bid.getPrice(), currency)
                : Money.valueOf(BigDecimal.ZERO, currency);

        Money maxAutobudgetBid = null;
        if (dbStrategy.getStrategyData().getBid() != null) {
            maxAutobudgetBid = Money.valueOf(dbStrategy.getStrategyData().getBid(), currency);
        }

        GeoTimezone geoTimezone = geoTimezoneByTimezoneId.get(campaign.getTimezoneId());
        LocalDateTime now = LocalDateTime.now(geoTimezone.getTimezone());
        TimeTarget timeTarget = campaign.getTimeTarget();
        double timeTargetCoef = 1.0;
        if (timeTarget != null) {
            ProductionCalendar calendar = productionCalendarByGeo.get(geoTimezone);
            timeTargetCoef = timeTarget.calcTimeTargetCoef(now, calendar);
        }

        AutoBrokerResult result = calculatePricesInternal(auctionStrategyMode, keywordBid, maxAutobudgetBid,
                bsResult.getPremium(), bsResult.getGuarantee(),
                timeTargetCoef,
                walletRestMoney.getRest());
        logger.trace("Calculated autobroker prices for keyword (id={}, price={}): {}",
                bid.getId(), keywordBid, result);
        return result;
    }

    static AutoBrokerResult calculatePricesInternal(
            AuctionStrategyMode strategyMode,
            Money keywordBid,
            @Nullable Money maxAutobudgetBid,
            BsCpcPrice[] premium,
            BsCpcPrice[] guarantee,
            double timeTargetCoef,
            Money effectiveRestMoney) {
        if (logger.isTraceEnabled()) {
            logger.trace("Calculating autobroker for params {strategy: {}, keywordBid: {}, maxAutobudgetBid: {}, "
                            + "premium: {}, guarantee: {}",
                    strategyMode, keywordBid, maxAutobudgetBid,
                    asList(premium), asList(guarantee)
            );
        }
        CurrencyCode currencyCode = keywordBid.getCurrencyCode();

        Function<BigDecimal, Money> moneyCreator = value -> Money.valueOf(value, currencyCode);

        Currency currency = Currencies.getCurrencies().get(currencyCode.name());
        Money minCurrencyPrice = moneyCreator.apply(currency.getMinPrice());

        Money price = keywordBid.multiply(timeTargetCoef);

        if (price.bigDecimalValue().compareTo(ZERO) == 0) {
            price = minCurrencyPrice;
        }

        if (maxAutobudgetBid != null) {
            price = MoneyUtils.min(price, maxAutobudgetBid);
        }

        Money effectivePrice;
        if (effectiveRestMoney.greaterThanZero() && effectiveRestMoney.lessThan(price)) {
            effectivePrice = effectiveRestMoney;
        } else {
            effectivePrice = price;
        }

        boolean truncated = false;
        BrokerPrice brokerPrice = null;
        Place place = null;

        LinkedHashMap<Place, AuctionPosition> shownPlacePrices =
                getAuctionBrokerPricesByPositions(currencyCode, premium, guarantee);

        // Интерфейс не использует 2-ую и 3-ю позиции в Гарантии, а API использует
        // Пока сделала так как нужно API, потом нужно будет учесть Интерфейс
        EnumSet<Place> shownPlaces = EnumSet.of(Place.PREMIUM1, Place.PREMIUM2, Place.PREMIUM3, Place.PREMIUM4,
                Place.GUARANTEE1, Place.GUARANTEE2, Place.GUARANTEE3, Place.GUARANTEE4);
        shownPlacePrices = EntryStream.of(shownPlacePrices)
                .filterKeys(shownPlaces::contains)
                .toCustomMap(LinkedHashMap::new);

        // или стратегия "наивысшая позиция в блоке"
        // или целились только в PREMIUM и не попали

        // если стратегия не рассматривает PREMIUM блок, то не рассматриваем соответствующие позиции
        AuctionPosition highestSearchingPlace =
                strategyMode.getTargetBlocks().contains(AuctionStrategyMode.Block.PREMIUM)
                        ? shownPlacePrices.get(Place.PREMIUM1)
                        : shownPlacePrices.get(Place.GUARANTEE1);
        Predicate<AuctionPosition> allowablePositions = p -> p.compareTo(highestSearchingPlace) >= 0;

        for (AuctionPosition auctionPosition : shownPlacePrices.values()) {
            if (!allowablePositions.test(auctionPosition)) {
                continue;
            }
            Money auctionPrice = auctionPosition.getBrokerPrice().getBidPrice();
            if (effectivePrice.lessThan(auctionPrice) && auctionPrice.lessThanOrEqual(price)) {
                truncated = true;
            } else if (auctionPrice.lessThanOrEqual(effectivePrice)) {
                brokerPrice = auctionPosition.getBrokerPrice();
                place = auctionPosition.getPlace();
                break;
            }
        }

        Money zero = moneyCreator.apply(ZERO);
        BrokerPrice zeroBrokerPrice = new BrokerPrice(zero, zero);


        double coverage = 1.0;
        if (brokerPrice == null) {
            // не удалось найти подходящее место в спец-размещении и блоке гарантированных показов
            // вычисляем покрытие
            // rotPrices (сейчас выпилен, но всегда был null), поэтому покрытие = 0
            coverage = 0.0;
            brokerPrice = zeroBrokerPrice;
            place = Place.ROTATION;
        }

        // обнуляем ставку на поиске, если она меньше минимальной
        if (brokerPrice.getBidPrice().lessThan(minCurrencyPrice)) {
            // ставка не может быть меньше шага торгов для текущей валюты
            brokerPrice = zeroBrokerPrice;
        } else {
            // Найдём минимальную ставку, при которой возможен показ
            Money effectiveMinPrice = Money.valueOfMicros(bsEffectiveMinPriceMicro(premium, guarantee), currencyCode);

            if (brokerPrice.getBidPrice().lessThan(effectiveMinPrice)) {
                brokerPrice = zeroBrokerPrice;
            }
        }

        Money brokerPriceRound = brokerPrice.getAmnestyPrice().roundToAuctionStepDown();

        Place placeWithoutCoef = place;
        if (timeTargetCoef != 1.0) {
            AutoBrokerResult autoBrokerResultNoCoef =
                    calculatePricesInternal(strategyMode, keywordBid, maxAutobudgetBid,
                            premium, guarantee, 1.0, effectiveRestMoney);
            placeWithoutCoef = autoBrokerResultNoCoef.getBrokerPlace();
        }

        return new AutoBrokerResult()
                .withBrokerPrice(brokerPriceRound)
                .withBrokerCoverage(coverage)
                .withBrokerTruncated(truncated)
                .withBrokerPlace(place)
                .withBrokerPlaceWithoutCoef(placeWithoutCoef);
    }

    /**
     * На основе ответа от Торгов БК определяем минимальную ставку, при которой возможен показ.
     * <p>
     * Формула такая:
     * {@code effectiveMinPrice = max(bs.minPrice, min(bs.premiumEnterPrice, min(bs.rotationEnterPrice, bs
     * .guaranteeEnterPrice)))}
     * NB: minPrice (сейчас выпилен, раньше сюда всегда приходил 0 для трафаретных торгов)
     */
    private static long bsEffectiveMinPriceMicro(BsCpcPrice[] premium, BsCpcPrice[] guarantee) {
        long premiumEnterPrice = premium[premium.length - 1].getPrice().micros();
        long sideShowsEnterPrice = guarantee[guarantee.length - 1].getPrice().micros();

        // У Спец-Размещения отдельные показатели CTR'ов, поэтому может оказаться, что для объявления
        // цена входа в Спец-Размещение ниже цены входа в Ротацию/Гарантию
        long showEnterPrice = Long.min(premiumEnterPrice, sideShowsEnterPrice);

        // подпираем минимальной ставкой, при которой возможен показ
        // скорее всего showEnterPrice не бывает отрицательным и этот max устарел
        return Long.max(0, showEnterPrice);
    }

    @Nonnull
    private static LinkedHashMap<Place, AuctionPosition> getAuctionBrokerPricesByPositions(
            CurrencyCode currencyCode, BsCpcPrice[] premium, BsCpcPrice[] guarantee) {
        LinkedHashMap<Place, AuctionPosition> shownPlacePrices = new LinkedHashMap<>();
        for (int i = 0; i < premium.length; i++) {
            BsCpcPrice placePrice = premium[i];
            Place p = Place.convertFromDb((long) (10 + i));
            shownPlacePrices.put(p,
                    new AuctionPosition(p,
                            placePrice.getPrice(),
                            placePrice.getCpc()));
        }

        for (int i = 0; i < guarantee.length; i++) {
            BsCpcPrice placePrice = guarantee[i];
            Place p = Place.convertFromDb((long) (20 + i));
            shownPlacePrices.put(p,
                    new AuctionPosition(p,
                            placePrice.getPrice(),
                            placePrice.getCpc()));
        }
        return shownPlacePrices;
    }

    private static AuctionStrategyMode getAuctionStrategyMode(DbStrategy dbStrategy) {
        if (dbStrategy.getAutobudget() != CampaignsAutobudget.NO) {
            return AuctionStrategyMode.HIGHEST_POSITION_ALL;
        }

        StrategyName strategyName = dbStrategy.getStrategyName();

        if (strategyName == StrategyName.NO_PREMIUM) {
            // Показ под результатами поиска на наивысшей позиции
            return AuctionStrategyMode.HIGHEST_POSITION_GUARANTEE;
        } else {
            // Наивысшая доступная позиция
            return AuctionStrategyMode.HIGHEST_POSITION_ALL;
        }
    }

}
