
package ru.yandex.direct.core.entity.autobroker.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.service.GeoTimezoneMappingService;
import ru.yandex.direct.core.entity.timetarget.service.ProductionCalendarProviderService;
import ru.yandex.direct.libs.timetarget.ProductionCalendar;

@ParametersAreNonnullByDefault
@Service
public class AutoBrokerCalculatorProviderService {

    private final ProductionCalendarProviderService productionCalendarProviderService;
    private final GeoTimezoneMappingService geoTimezoneMappingService;

    @Autowired
    public AutoBrokerCalculatorProviderService(
            ProductionCalendarProviderService productionCalendarProviderService,
            GeoTimezoneMappingService geoTimezoneMappingService) {
        this.productionCalendarProviderService = productionCalendarProviderService;
        this.geoTimezoneMappingService = geoTimezoneMappingService;
    }

    /**
     * @return {@link AutoBrokerCalculator} для вычисления цен автоброкера для ставок,
     * относящихся к заданному набору кампаний.
     */
    public AutoBrokerCalculator getAutoBrokerCalculatorForCampaigns(Collection<Campaign> campaigns) {
        Map<Long, GeoTimezone> geoTimezoneByTimezoneId =
                StreamEx.of(campaigns)
                        .map(Campaign::getTimezoneId)
                        .distinct()
                        .mapToEntry(geoTimezoneMappingService::getRegionIdByTimezoneId).toMap();

        Map<GeoTimezone, ProductionCalendar> productionCalendarByGeo =
                StreamEx.ofValues(geoTimezoneByTimezoneId)
                        .distinct()
                        .mapToEntry(geoTimezone -> {
                            LocalDateTime now = LocalDateTime.now(geoTimezone.getTimezone());
                            return productionCalendarProviderService
                                    .getProductionCalendar(now.getYear(), geoTimezone.getRegionId());
                        })
                        .toMap();

        Set<Long> expectedCampaignIds = StreamEx.of(campaigns).map(Campaign::getId).toSet();

        return new AutoBrokerCalculator(expectedCampaignIds, geoTimezoneByTimezoneId, productionCalendarByGeo);
    }

}
