package ru.yandex.direct.core.entity.autobudget.repository;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.Optional;
import java.util.Timer;

import javax.annotation.Nonnull;
import javax.annotation.PreDestroy;

import ru.yandex.direct.ytwrapper.client.YtExecutionUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.selector.ClusterFreshness;
import ru.yandex.direct.ytwrapper.dynamic.selector.WeightBasedClusterChooser;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.utils.DateTimeUtils.MOSCOW_TIMEZONE;

abstract public class AbstractAutobudgetYtRepository {
    private static final Duration FRESH_DELAY = Duration.ofHours(2);
    private static final Duration YT_CLUSTER_REFRESH_INTERVAL = Duration.ofMinutes(5);

    protected final YtProvider ytProvider;
    private final Timer timer;
    private final WeightBasedClusterChooser clusterChooser;

    public AbstractAutobudgetYtRepository(YtProvider ytProvider,
                                          CpaAutobudgetClusterConfig config) {
        this.ytProvider = ytProvider;
        timer = new Timer(getClass().getSimpleName() + "-Times", true);
        clusterChooser = new WeightBasedClusterChooser(
                timer,
                YT_CLUSTER_REFRESH_INTERVAL,
                config.getClusters(),
                (cluster) -> clusterFreshness(cluster, getTable()));
    }

    @PreDestroy
    public void stopTimer() {
        if (timer != null) {
            timer.cancel();
        }
    }

    private ClusterFreshness clusterFreshness(YtCluster cluster, YtTable table) {
        Long tableUpdateTime = getUpdateUnixTime(cluster, table);
        long now = System.currentTimeMillis() / 1000L;
        if (tableUpdateTime + FRESH_DELAY.getSeconds() > now) {
            return ClusterFreshness.FRESH;
        } else {
            return ClusterFreshness.ROTTEN;
        }
    }

    private Long getUpdateUnixTime(YtCluster cluster, YtTable table) {
        String tableUpdateTime = ytProvider.getOperator(cluster).readTableModificationTime(table);
        return Instant.parse(tableUpdateTime).getEpochSecond();
    }

    protected abstract YtTable getTable();

    /**
     * Возвращает список доступных кластеров в том порядке, в котором к ним необходимо обращаться
     */
    @Nonnull
    public Optional<List<YtCluster>> getClustersOrdered() {
        return clusterChooser.getClustersOrdered();
    }

    /**
     * Возвращает список доступных кластеров в том порядке, в котором к ним необходимо обращаться
     *
     * @throws IllegalStateException если нет доступных кластеров
     */
    @Nonnull
    public List<YtCluster> getAvailableClustersOrdered() {
        return getClustersOrdered().orElseThrow(() -> new IllegalStateException("No any clusters"));
    }

    public LocalDateTime getTableLastUpdateTime() {
        List<YtCluster> clustersByPriority = getAvailableClustersOrdered();
        String modificationTime = YtExecutionUtil.executeWithFallback(clustersByPriority, ytProvider::getOperator,
                operator -> operator.readTableModificationTime(getTable()));
        return LocalDateTime.ofInstant(Instant.parse(modificationTime), ZoneId.of(MOSCOW_TIMEZONE));
    }
}
