package ru.yandex.direct.core.entity.autobudget.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.autobudget.model.CpaAutobudgetAlert;
import ru.yandex.direct.dbschema.ppc.enums.AutobudgetCpaAlertsStatus;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.Tables.AUTOBUDGET_CPA_ALERTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Работа с CPA-алертами автобюджета
 * Таблица: ppc.autobudget_cpa_alerts
 */
@Repository
@ParametersAreNonnullByDefault
public class AutobudgetCpaAlertRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<CpaAutobudgetAlert> alertMapper;

    @Autowired
    public AutobudgetCpaAlertRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        alertMapper = JooqMapperWithSupplierBuilder.builder(CpaAutobudgetAlert::new)
                .map(property(CpaAutobudgetAlert.CID, AUTOBUDGET_CPA_ALERTS.CID))
                .map(property(CpaAutobudgetAlert.CPA_DEVIATION, AUTOBUDGET_CPA_ALERTS.CPA_DEVIATION))
                .map(property(CpaAutobudgetAlert.APC_DEVIATION, AUTOBUDGET_CPA_ALERTS.APC_DEVIATION))
                .map(property(CpaAutobudgetAlert.LAST_UPDATE, AUTOBUDGET_CPA_ALERTS.LAST_UPDATE))
                .map(convertibleProperty(CpaAutobudgetAlert.STATUS, AUTOBUDGET_CPA_ALERTS.STATUS,
                        AutobudgetMapping::cpaAlertsStatusFromDb,
                        AutobudgetMapping::cpaAlertsStatusToDb))
                .build();
    }

    /**
     * Сохраняет список CPA-алертов в таблице ppc.autobudget_cpa_alerts
     *
     * @param shard  шард
     * @param alerts список алертов
     */
    public void addAlerts(int shard, Collection<CpaAutobudgetAlert> alerts) {
        if (alerts.isEmpty()) {
            return;
        }
        new InsertHelper<>(dslContextProvider.ppc(shard), AUTOBUDGET_CPA_ALERTS)
                .addAll(alertMapper, alerts)
                .execute();
    }

    /**
     * Удалить активные CPA-алерты, внесенные до заданного времени с указанным статусом
     *
     * @param borderDateTime Дата и время, основываясь на которых нужно удалять записи
     * @return количество записей, удаленных из таблицы
     */
    public int deleteActiveAlertsOlderThanDateTime(int shard, LocalDateTime borderDateTime) {
        return deleteAlertsOlderThanDateTime(shard, AutobudgetCpaAlertsStatus.active, borderDateTime);
    }

    /**
     * Удалить из таблицы ppc.autobudget_cpa_alerts все записи, внесенные до заданного времени с указанным статусом
     *
     * @param alertsStatus   Статус алерта, которые хотим удалить
     * @param borderDateTime Дата и время, основываясь на которых нужно удалять записи
     * @return количество записей, удаленных из таблицы
     */
    @QueryWithoutIndex("Чистка таблицы, выполняется в jobs")
    private int deleteAlertsOlderThanDateTime(int shard, AutobudgetCpaAlertsStatus alertsStatus,
                                              LocalDateTime borderDateTime) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(AUTOBUDGET_CPA_ALERTS)
                .where(AUTOBUDGET_CPA_ALERTS.STATUS.eq(alertsStatus)
                        .and(AUTOBUDGET_CPA_ALERTS.LAST_UPDATE.lessThan(borderDateTime)))
                .execute();
    }

    /**
     * Метод для получения id кампаний, у которых есть алерты в таблице ppc.autobudget_cpa_alerts
     *
     * @param campaignIds список id кампаний, для которых выполняется поиск
     * @return id кампаний, у которых есть алерты
     */
    public Set<Long> getCidOfExistingAlerts(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(AUTOBUDGET_CPA_ALERTS.CID)
                .from(AUTOBUDGET_CPA_ALERTS)
                .where(AUTOBUDGET_CPA_ALERTS.CID.in(campaignIds))
                .fetchSet(AUTOBUDGET_CPA_ALERTS.CID);
    }

    /**
     * Метод для заморозки алертов в таблице ppc.autobudget_cpa_alerts
     *
     * @param shard       шард
     * @param campaignIds список кампаний, для которых замораживаются алерты
     */
    public void freezeAlerts(int shard, Collection<Long> campaignIds) {
        if (!campaignIds.isEmpty()) {
            freezeAlerts(dslContextProvider.ppc(shard), campaignIds);
        }
    }

    public void freezeAlerts(DSLContext dslContext, Collection<Long> campaignIds) {
        if (!campaignIds.isEmpty()) {
            dslContext
                    .update(AUTOBUDGET_CPA_ALERTS)
                    .set(AUTOBUDGET_CPA_ALERTS.STATUS, AutobudgetCpaAlertsStatus.frozen)
                    .set(AUTOBUDGET_CPA_ALERTS.LAST_UPDATE, LocalDateTime.now())
                    .where(AUTOBUDGET_CPA_ALERTS.CID.in(campaignIds))
                    .execute();
        }
    }

    /**
     * Получение автобюджетной проблемы по номерам кампаний
     *
     * @param shard       номер шарда
     * @param campaignIds номера кампаний
     * @return мапа номера кампании на автобюджетную проблему
     * (сожержит только те id кампаний, для которых есть проблемы)
     */
    public Map<Long, CpaAutobudgetAlert> getAlerts(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(alertMapper.getFieldsToRead())
                .from(AUTOBUDGET_CPA_ALERTS)
                .where(AUTOBUDGET_CPA_ALERTS.CID.in(campaignIds))
                .fetchMap(rec -> rec.getValue(AUTOBUDGET_CPA_ALERTS.CID), alertMapper::fromDb);
    }
}
