package ru.yandex.direct.core.entity.autobudget.repository;

import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.autobudget.model.AutobudgetForecast;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.Tables.AUTOBUDGET_FORECAST;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Работа с данными прогноза для кампаний с автобюджетом
 * Таблица: ppc.autobudget_forecast
 */
@Repository
@ParametersAreNonnullByDefault
public class AutobudgetForecastRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<AutobudgetForecast> mapper;

    @Autowired
    public AutobudgetForecastRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        mapper = JooqMapperWithSupplierBuilder.builder(AutobudgetForecast::new)
                .map(property(AutobudgetForecast.CID, AUTOBUDGET_FORECAST.CID))
                .map(property(AutobudgetForecast.AUTOBUDGET_FORECAST_DATE, AUTOBUDGET_FORECAST.AUTOBUDGET_FORECAST_DATE))
                .map(property(AutobudgetForecast.AUTOBUDGET_FORECAST, AUTOBUDGET_FORECAST.AUTOBUDGET_FORECAST_))
                .map(property(AutobudgetForecast.AUTOBUDGET_FORECAST_CLICKS, AUTOBUDGET_FORECAST.AUTOBUDGET_FORECAST_CLICKS))
                .map(convertibleProperty(AutobudgetForecast.STATUS_AUTOBUDGET_FORECAST, AUTOBUDGET_FORECAST.STATUS_AUTOBUDGET_FORECAST,
                        AutobudgetMapping::autobudgetForecastStatusFromDb,
                        AutobudgetMapping::autobudgetForecastStatusToDb))
                .build();
    }

    public void addAutobudgetForecast(int shard, Collection<AutobudgetForecast> forecasts) {
        if (forecasts.isEmpty()) {
            return;
        }

        var insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), AUTOBUDGET_FORECAST)
                .addAll(mapper, forecasts);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(AUTOBUDGET_FORECAST.CID, MySQLDSL.values(AUTOBUDGET_FORECAST.CID));
            insertHelper.execute();
        }
    }

    /**
     * Удалить прогноз автобюджета для кампании
     *
     * @return количество записей, удаленных из таблицы
     */
    public int delete(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(AUTOBUDGET_FORECAST)
                .where(AUTOBUDGET_FORECAST.CID.in(campaignIds))
                .execute();
    }
}
