package ru.yandex.direct.core.entity.autobudget.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.autobudget.model.HourlyAutobudgetAlert;
import ru.yandex.direct.dbschema.ppc.enums.AutobudgetAlertsStatus;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.Tables.AUTOBUDGET_ALERTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Работа с часовыми алертами автобюджета
 * Таблица: ppc.autobudget_alerts
 */
@Repository
@ParametersAreNonnullByDefault
public class AutobudgetHourlyAlertRepository {

    private final JooqMapperWithSupplier<HourlyAutobudgetAlert> jooqMapper;
    private final Set<Field<?>> fieldsToRead;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public AutobudgetHourlyAlertRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        jooqMapper = JooqMapperWithSupplierBuilder.builder(HourlyAutobudgetAlert::new)
                .map(property(HourlyAutobudgetAlert.CID, AUTOBUDGET_ALERTS.CID))
                .map(convertibleProperty(HourlyAutobudgetAlert.PROBLEMS, AUTOBUDGET_ALERTS.PROBLEMS,
                        AutobudgetMapping::problemsFromDb,
                        AutobudgetMapping::problemsToDb))
                .map(property(HourlyAutobudgetAlert.OVERDRAFT, AUTOBUDGET_ALERTS.OVERDRAFT))
                .map(property(HourlyAutobudgetAlert.LAST_UPDATE, AUTOBUDGET_ALERTS.LAST_UPDATE))
                .map(convertibleProperty(HourlyAutobudgetAlert.STATUS, AUTOBUDGET_ALERTS.STATUS,
                        AutobudgetMapping::hourlyAlertsStatusFromDb,
                        AutobudgetMapping::hourlyAlertsStatusToDb))
                .build();
        fieldsToRead = jooqMapper.getFieldsToRead();
    }

    /**
     * Сохраняет список часовых алертов в таблице ppc.autobudget_alerts
     *
     * @param shard  шард
     * @param alerts список алертов
     */
    public void addAlerts(int shard, Collection<HourlyAutobudgetAlert> alerts) {
        new InsertHelper<>(dslContextProvider.ppc(shard), AUTOBUDGET_ALERTS)
                .addAll(jooqMapper, alerts)
                .executeIfRecordsAdded();
    }

    /**
     * Удалить активные алерты, внесенные до заданного времени с указанным статусом
     *
     * @param borderDateTime Дата и время, основываясь на которых нужно удалять записи
     * @return количество записей, удаленных из таблицы
     */
    public int deleteActiveAlertsOlderThanDateTime(int shard, LocalDateTime borderDateTime) {
        return deleteAlertsOlderThanDateTime(shard, AutobudgetAlertsStatus.active, borderDateTime);
    }

    /**
     * Удалить из таблицы ppc.autobudget_alerts все записи, внесенные до заданного времени с указанным статусом
     *
     * @param alertsStatus   Статус алерта, которые хотим удалить
     * @param borderDateTime Дата и время, основываясь на которых нужно удалять записи
     * @return количество записей, удаленных из таблицы
     */
    @QueryWithoutIndex("Чистка таблицы, выполняется в jobs")
    private int deleteAlertsOlderThanDateTime(int shard, AutobudgetAlertsStatus alertsStatus,
                                              LocalDateTime borderDateTime) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(AUTOBUDGET_ALERTS)
                .where(AUTOBUDGET_ALERTS.STATUS.eq(alertsStatus)
                        .and(AUTOBUDGET_ALERTS.LAST_UPDATE.lessThan(borderDateTime)))
                .execute();
    }

    /**
     * Метод для получения id кампаний, у которых есть алерты в таблице ppc.autobudget_alerts
     *
     * @param campaignIds список id кампаний, для которых выполняется поиск
     * @return id кампаний, у которых есть алерты
     */
    public Set<Long> getCidOfExistingAlerts(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(AUTOBUDGET_ALERTS.CID)
                .from(AUTOBUDGET_ALERTS)
                .where(AUTOBUDGET_ALERTS.CID.in(campaignIds))
                .fetchSet(AUTOBUDGET_ALERTS.CID);
    }

    /**
     * Получение автобюджетной проблемы по номерам кампаний
     *
     * @param shard       номер шарда
     * @param campaignIds номера кампаний
     * @return мапа номера кампании на автобюджетную проблему
     * (сожержит только те id кампаний, для которых есть проблемы)
     */
    public Map<Long, HourlyAutobudgetAlert> getAlerts(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(AUTOBUDGET_ALERTS)
                .where(AUTOBUDGET_ALERTS.CID.in(campaignIds))
                .fetchMap(rec -> rec.getValue(AUTOBUDGET_ALERTS.CID), jooqMapper::fromDb);
    }

    /**
     * Перевод автобюджетных проблем на кампаниях в замороженный режим
     *
     * @param shard       номер шарда
     * @param campaignIds номера кампаний
     */
    public void freezeAlerts(int shard, Collection<Long> campaignIds) {
        if (!campaignIds.isEmpty()) {
            freezeAlerts(dslContextProvider.ppc(shard), campaignIds);
        }
    }

    public void freezeAlerts(DSLContext dslContext, Collection<Long> campaignIds) {
        if (!campaignIds.isEmpty()) {
            dslContext
                    .update(AUTOBUDGET_ALERTS)
                    .set(AUTOBUDGET_ALERTS.STATUS, AutobudgetAlertsStatus.frozen)
                    .set(AUTOBUDGET_ALERTS.LAST_UPDATE, LocalDateTime.now())
                    .where(AUTOBUDGET_ALERTS.CID.in(campaignIds))
                    .execute();
        }
    }
}
