package ru.yandex.direct.core.entity.autobudget.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.autobudget.model.CpaAutobudgetPessimizedUser;
import ru.yandex.direct.dbschema.ppcdict.tables.records.CpaAutobudgetPessimizedUsersRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.CpaAutobudgetPessimizedUsers.CPA_AUTOBUDGET_PESSIMIZED_USERS;

@Repository
@ParametersAreNonnullByDefault
public class CpaAutobudgetPessimizedUsersRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<CpaAutobudgetPessimizedUser> jooqMapper;

    @Autowired
    public CpaAutobudgetPessimizedUsersRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(CpaAutobudgetPessimizedUser::new)
                .map(ReaderWriterBuilders.property(CpaAutobudgetPessimizedUser.LOGIN,
                        CPA_AUTOBUDGET_PESSIMIZED_USERS.LOGIN))
                .map(ReaderWriterBuilders.property(CpaAutobudgetPessimizedUser.CLIENT_ID,
                        CPA_AUTOBUDGET_PESSIMIZED_USERS.CLIENT_ID))
                .map(ReaderWriterBuilders.property(CpaAutobudgetPessimizedUser.MIN_BID_TARGET_CPA_MULTIPLIER,
                        CPA_AUTOBUDGET_PESSIMIZED_USERS.MIN_BID_TARGET_CPA_MULTIPLIER))
                .build();
    }

    public int addOrUpdatePessimizedUsers(Collection<CpaAutobudgetPessimizedUser> pessimizedLogins) {
        InsertHelper<CpaAutobudgetPessimizedUsersRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(),
                        CPA_AUTOBUDGET_PESSIMIZED_USERS)
                        .addAll(jooqMapper, pessimizedLogins);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CPA_AUTOBUDGET_PESSIMIZED_USERS.MIN_BID_TARGET_CPA_MULTIPLIER,
                            MySQLDSL.values(CPA_AUTOBUDGET_PESSIMIZED_USERS.MIN_BID_TARGET_CPA_MULTIPLIER))
                    .set(CPA_AUTOBUDGET_PESSIMIZED_USERS.UPDATE_TIME,
                            MySQLDSL.values(CPA_AUTOBUDGET_PESSIMIZED_USERS.UPDATE_TIME));
        }
        return insertHelper.executeIfRecordsAdded();
    }

    public int deletePessimizedUsersOlderThan(LocalDateTime boundaryDateTime) {
        return dslContextProvider.ppcdict()
                .deleteFrom(CPA_AUTOBUDGET_PESSIMIZED_USERS)
                .where(CPA_AUTOBUDGET_PESSIMIZED_USERS.UPDATE_TIME.lessThan(boundaryDateTime))
                .execute();
    }

    public Optional<CpaAutobudgetPessimizedUser> getPessimizedClientId(Long clientId) {
        return StreamEx.of(getPessimizedUsersByClientId(List.of(clientId)))
                .findAny();
    }

    public List<CpaAutobudgetPessimizedUser> getPessimizedUsersByClientId(Collection<Long> clientIds) {
        if (clientIds.isEmpty()) {
            return Collections.emptyList();
        }
        return dslContextProvider.ppcdict()
                .select(jooqMapper.getFieldsToRead())
                .from(CPA_AUTOBUDGET_PESSIMIZED_USERS)
                .where(CPA_AUTOBUDGET_PESSIMIZED_USERS.CLIENT_ID.in(clientIds))
                .fetch(jooqMapper::fromDb);
    }
}
