package ru.yandex.direct.core.entity.autobudget.repository;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.autobudget.model.CpaAutobudgetPessimizedUser;
import ru.yandex.direct.core.entity.autobudget.ytmodels.generated.YtPessimizedLoginsRow;
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.core.entity.autobudget.ytmodels.generated.YtDbTables.PESSIMIZEDLOGINS;
import static ru.yandex.direct.utils.DateTimeUtils.MOSCOW_TIMEZONE;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Repository
@ParametersAreNonnullByDefault
public class CpaAutobudgetPessimizedUsersYtRepository extends AbstractAutobudgetYtRepository {

    @Autowired
    public CpaAutobudgetPessimizedUsersYtRepository(YtProvider ytProvider,
                                                    CpaAutobudgetClusterConfig config) {
        super(ytProvider, config);
    }

    @Override
    protected YtTable getTable() {
        return PESSIMIZEDLOGINS;
    }

    /**
     * Вычитывает данные из таблицы с пессимизированными логинами чанками размером {@code chunkSize} и передает их в
     * {@code addOrUpdateFunction} для последующей обработки.
     *
     * @param addOrUpdateFunction функция для обработки прочитанных данных
     * @param chunkSize           кол-во строк которые нужно прочитать за раз и передать в {@code addOrUpdateFunction}
     * @return количество прочитанных строк
     */
    public Integer getAndApplyPessimizedLogins(Function<List<CpaAutobudgetPessimizedUser>, Integer> addOrUpdateFunction, int chunkSize) {
        final AtomicInteger changedCount = new AtomicInteger();

        List<YtCluster> clustersByPriority = getAvailableClustersOrdered();

        YtExecutionUtil.executeWithFallback(clustersByPriority,
                ytProvider::getOperator,
                operator -> operator.readTableSnapshot(PESSIMIZEDLOGINS, new YtPessimizedLoginsRow(),
                        CpaAutobudgetPessimizedUsersYtRepository::toCpaAutobudgetPessimizedUser,
                        r -> addAndGet(addOrUpdateFunction, changedCount, r), chunkSize));
        return changedCount.get();
    }

    private int addAndGet(Function<List<CpaAutobudgetPessimizedUser>, Integer> addOrUpdateFunction,
                          AtomicInteger changedCount, List<CpaAutobudgetPessimizedUser> r) {
        var filledPessimizedUsers = filterList(r, Objects::nonNull);
        return changedCount.addAndGet(addOrUpdateFunction.apply(filledPessimizedUsers));
    }

    @Nullable
    public static CpaAutobudgetPessimizedUser toCpaAutobudgetPessimizedUser(YtPessimizedLoginsRow row) {
        if (row.getLogin() == null) {
            return null;
        }
        return new CpaAutobudgetPessimizedUser()
                .withClientId(null)
                .withLogin(row.getLogin())
                .withMinBidTargetCpaMultiplier(getMinBidTargetCpaMultiplier(row.getMinBidTargetCpaMultiplier()))
                .withUpdateTime(LocalDateTime.ofInstant(Instant.ofEpochSecond(row.getUpdateTimestamp()),
                        ZoneId.of(MOSCOW_TIMEZONE)));
    }

    private static BigDecimal getMinBidTargetCpaMultiplier(@Nullable Double minBidTargetCpaMultiplier) {
        if (minBidTargetCpaMultiplier == null) {
            return BigDecimal.ONE;
        }
        return BigDecimal.valueOf(minBidTargetCpaMultiplier);
    }

}
