package ru.yandex.direct.core.entity.autobudget.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem;
import ru.yandex.direct.core.entity.autobudget.model.AutobudgetCommonAlertStatus;
import ru.yandex.direct.core.entity.autobudget.model.AutobudgetHourlyProblem;
import ru.yandex.direct.core.entity.autobudget.model.HourlyAutobudgetAlert;
import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetCpaAlertRepository;
import ru.yandex.direct.core.entity.autobudget.repository.AutobudgetHourlyAlertRepository;
import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithFreezingStrategyAlerts;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithNetworkSettings;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.TextCampaignWithCustomStrategy;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem.ENGINE_MIN_COST_LIMITED;
import static ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem.MAX_BID_REACHED;
import static ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem.NO_PROBLEM;
import static ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem.UPPER_POSITIONS_REACHED;
import static ru.yandex.direct.core.entity.autobudget.model.AutobudgetAggregatedHourlyProblem.WALLET_DAY_BUDGET_REACHED;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.AUTO_CONTEXT_LIMIT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CONTEXT_LIMIT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.SHOWS_DISABLED_CONTEXT_LIMIT;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для работы с предупреждениями автобюджета на кампании, которые мы получаем от БК
 * Предупреждения сигнализируют о каких-то проблемах в настройках рекламной кампании, которые
 * не позволяют автобюджету выполнять поставленные ему задачи (выдержать среднюю цену/недельный бюджет/...)
 * Некоторые из этих предупреждений мы показываем в интерфейсе рекламодателю, чтобы он мог принять меры
 * для устранения препятствий. Например поднять дневной бюджет на общем счете.
 */
@ParametersAreNonnullByDefault
@Service
public class AutobudgetAlertService {
    private final AutobudgetHourlyAlertRepository autobudgetHourlyAlertRepository;
    private final ShardHelper shardHelper;
    private final AutobudgetCpaAlertRepository autobudgetCpaAlertRepository;

    @Autowired
    public AutobudgetAlertService(
            AutobudgetHourlyAlertRepository autobudgetHourlyAlertRepository,
            ShardHelper shardHelper, AutobudgetCpaAlertRepository autobudgetCpaAlertRepository) {
        this.autobudgetHourlyAlertRepository = autobudgetHourlyAlertRepository;
        this.shardHelper = shardHelper;
        this.autobudgetCpaAlertRepository = autobudgetCpaAlertRepository;
    }

    /**
     * "Замораживание" автобюджетных предупреждений на некоторое время для ключевых слов и динамических условий.
     * Это нужно чтобы временно перестать их показывать, так как они могут уже быть не актуальными.
     * <p>
     * Замораживает только предупреждения связанные с ключевыми словами или динамическими условиями.
     *
     * @param campaignIds id активных кампаний, в которых изменились ключевые фразы
     */
    public void freezeAlertsOnKeywordsChange(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientId(clientId);

        Map<Long, AutobudgetAggregatedHourlyProblem> aggregatedProblems =
                getAggregatedHourlyProblems(shard, campaignIds);

        Predicate<AutobudgetAggregatedHourlyProblem> problemNeedsFreeze = problem ->
                problem == MAX_BID_REACHED || problem == UPPER_POSITIONS_REACHED;

        Set<Long> campaignIdsToFreeze = EntryStream.of(aggregatedProblems)
                .filterValues(problemNeedsFreeze)
                .keys()
                .toSet();
        autobudgetHourlyAlertRepository.freezeAlerts(shard, campaignIdsToFreeze);
    }

    /**
     * Метод обновлению автобюджетных подсказок на случай смены ДРФ.
     * Замораживаем подсказки автобюджета при наличии подсказок MAX_BID_REACHED
     * или UPPER_POSITIONS_REACHED и отключении ДРФ.
     *
     * @param clientId  - id клиента
     * @param campaigns - изменение моделей кампаний
     */
    public void freezeAlertsOnBroadMatchChange(ClientId clientId,
                                               Collection<AppliedChanges<CampaignWithBroadMatch>> campaigns) {
        AlertsFreezeInfo info = getCampaignsToFreezeAlertsOnBroadMatchChange(clientId, campaigns);

        int shard = shardHelper.getShardByClientId(clientId);
        autobudgetHourlyAlertRepository.freezeAlerts(shard, info.getCampaignsToFreezeHourlyAlerts());
    }

    public AlertsFreezeInfo getCampaignsToFreezeAlertsOnBroadMatchChange(ClientId clientId,
                                                                         Collection<AppliedChanges<CampaignWithBroadMatch>> campaigns) {
        var broadMatchingDisabled = campaigns
                .stream()
                .filter(ac -> {
                    BroadMatch oldBroadMatch = ac.getOldValue(CampaignWithBroadMatch.BROAD_MATCH);
                    BroadMatch newBroadMatch = ac.getNewValue(CampaignWithBroadMatch.BROAD_MATCH);
                    int oldValueLimit = oldBroadMatch.getBroadMatchLimit();
                    int newValueLimit = newBroadMatch.getBroadMatchLimit();
                    boolean broadMatchFlag = newBroadMatch.getBroadMatchFlag();
                    return !broadMatchFlag
                            || oldValueLimit > 0 && (newValueLimit == -1 || oldValueLimit < newValueLimit);
                })
                .map(ac -> ac.getModel().getId())
                .collect(Collectors.toSet());

        var campaignsToFreeze = EntryStream.of(getAggregatedHourlyProblems(clientId, broadMatchingDisabled))
                .filterValues(problem -> problem == MAX_BID_REACHED || problem == UPPER_POSITIONS_REACHED)
                .keys()
                .toSet();
        return new AlertsFreezeInfo(campaignsToFreeze);
    }

    /**
     * Получение агрегированных автобюджетных проблем для указанных кампаний на основе тех, которые пришли из БК.
     * Эта проблема отображается в интерфейсе, чтобы сподвигнуть рекламодателя поправить настройки своей кампании.
     * Мы отображаем не все возможные проблемы, и если у кампании есть одновременно несколько проблем, из них
     * выбирается наиболее приоритетная.
     * Например, если автобюджет говорит, что достигнута максимальная ставка из параметров стратегии, и что
     * дневной бюджет на общем счету слишком маленький, то мы показываем предупреждение про дневной бюджет, т.к.
     * мы считаем, что эту проблему нужно решить в первую очередь.
     *
     * @param campaignIds id кампаний
     * @return мапа id кампаний на их самые приоритетные проблем из имеющихся автобюджетных проблем,
     * или AutobudgetAggregatedHourlyProblem.NO_PROBLEM, если проблем нет.
     */
    public Map<Long, AutobudgetAggregatedHourlyProblem> getAggregatedHourlyProblems(
            ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getAggregatedHourlyProblems(shard, campaignIds);
    }

    private Map<Long, AutobudgetAggregatedHourlyProblem> getAggregatedHourlyProblems(
            int shard, Collection<Long> campaignIds) {
        Map<Long, HourlyAutobudgetAlert> alerts = autobudgetHourlyAlertRepository.getAlerts(shard, campaignIds);
        return StreamEx.of(campaignIds)
                .mapToEntry(campaignId -> this.toAggregatedProblem(alerts.get(campaignId)))
                .toMap();
    }

    public Map<Long, AutobudgetAggregatedHourlyProblem> getActiveHourlyProblemsAlerts(
            int shard, Collection<Long> campaignIds) {
        Map<Long, HourlyAutobudgetAlert> alerts = autobudgetHourlyAlertRepository.getAlerts(shard, campaignIds);
        return EntryStream.of(alerts)
                .filterValues(alert -> alert.getStatus() == AutobudgetCommonAlertStatus.ACTIVE)
                .mapValues(this::toAggregatedProblem)
                .toMap();
    }

    /**
     * Конвертация автобюджетных проблем, которые приходят из БК в агрегированную форму,
     * которая показывается рекламодателю, чтобы сподвигнуть его исправить настройки своей кампании.
     * Если у кампании есть одновременно несколько проблем, из них выбирается наиболее приоритетная.
     *
     * @param autobudgetAlert объект с автобюджетными проблемами
     * @return агрегированное представление проблемы
     */
    private AutobudgetAggregatedHourlyProblem toAggregatedProblem(@Nullable HourlyAutobudgetAlert autobudgetAlert) {
        AutobudgetAggregatedHourlyProblem result = NO_PROBLEM;
        if (autobudgetAlert == null) {
            return result;
        }

        Set<AutobudgetHourlyProblem> problems = autobudgetAlert.getProblems();
        if (problems.contains(AutobudgetHourlyProblem.WALLET_DAILY_BUDGET_REACHED)) {
            result = WALLET_DAY_BUDGET_REACHED;
        } else if (problems.contains(AutobudgetHourlyProblem.ENGINE_MIN_COST_LIMITED)) {
            result = ENGINE_MIN_COST_LIMITED;
        } else if (problems.contains(AutobudgetHourlyProblem.UPPER_POSITIONS_REACHED)) {
            result = UPPER_POSITIONS_REACHED;
        } else if (problems.contains(AutobudgetHourlyProblem.MAX_BID_REACHED) ||
                problems.contains(AutobudgetHourlyProblem.MARGINAL_PRICE_REACHED)) {
            result = MAX_BID_REACHED;
        }
        return result;
    }

    public static boolean isChanged(@Nullable DbStrategy firstStrategy, @Nullable DbStrategy secondStrategy,
                                    @Nullable ModelProperty<? super StrategyData, ?> property) {
        if ((firstStrategy == null) != (secondStrategy == null)) {
            return true;
        } else {
            if (firstStrategy == null) {
                return false;
            }
            if (property == null) {
                return firstStrategy.getStrategyName() != secondStrategy.getStrategyName()
                        || firstStrategy.getPlatform() != secondStrategy.getPlatform();
            }
            BigDecimal prop1 = (BigDecimal) property.get(firstStrategy.getStrategyData());
            BigDecimal prop2 = (BigDecimal) property.get(secondStrategy.getStrategyData());
            if (prop1 == null && prop2 == null) {
                return false;
            }
            return prop1 == null || prop2 == null || prop1.compareTo(prop2) != 0;
        }
    }

    /**
     * аналог перлового  update_on_strategy_change
     */
    public void freezeAlertsOnStrategyChange(ClientId clientId,
                                             Collection<AppliedChanges<TextCampaignWithCustomStrategy>> campaigns) {
        AlertsFreezeInfo info = getCampaignsToFreezeAlertsOnStrategyChange(clientId, campaigns);

        // замораживаем, все что нужно
        int shard = shardHelper.getShardByClientId(clientId);
        autobudgetCpaAlertRepository.freezeAlerts(shard, info.getCampaignsToFreezeCpaAlerts());
        autobudgetHourlyAlertRepository.freezeAlerts(shard, info.getCampaignsToFreezeHourlyAlerts());
    }

    public <C extends CampaignWithFreezingStrategyAlerts> AlertsFreezeInfo getCampaignsToFreezeAlertsOnStrategyChange(
            ClientId clientId,
            Collection<AppliedChanges<C>> campaigns
    ) {
        List<Long> campaignsIds = mapList(campaigns, c -> c.getModel().getId());

        Map<Long, AutobudgetAggregatedHourlyProblem> aggregatedHourlyProblems = getAggregatedHourlyProblems(clientId,
                campaignsIds);

        int shard = shardHelper.getShardByClientId(clientId);
        Set<Long> cidsOfExistingCpaAlerts = autobudgetCpaAlertRepository.getCidOfExistingAlerts(shard, campaignsIds);
        Set<Long> toFreezeHourlyAlerts = new HashSet<>();
        Set<Long> toFreezeCpaAlerts = new HashSet<>();
        for (var ac : campaigns) {
            Long campaignId = ac.getModel().getId();
            DbStrategy oldStrategy = ac.getOldValue(CampaignWithFreezingStrategyAlerts.STRATEGY);
            DbStrategy newStrategy = ac.getNewValue(CampaignWithFreezingStrategyAlerts.STRATEGY);
            AutobudgetAggregatedHourlyProblem problem = aggregatedHourlyProblems.get(campaignId);
            if (problem != NO_PROBLEM && (newStrategy.isAutoBudget() || oldStrategy.isAutoBudget())) {

                if (shouldAlertsBeFrozen(problem, oldStrategy, newStrategy)) {
                    toFreezeHourlyAlerts.add(campaignId);
                }
                if (cidsOfExistingCpaAlerts.contains(campaignId) && hasAvgCpaChanged(oldStrategy, newStrategy)) {
                    toFreezeCpaAlerts.add(campaignId);
                }
            }
        }
        return new AlertsFreezeInfo(toFreezeHourlyAlerts, toFreezeCpaAlerts);
    }

    private boolean shouldAlertsBeFrozen(AutobudgetAggregatedHourlyProblem problem,
                                         @Nullable DbStrategy oldStrategy, @Nullable DbStrategy newStrategy) {
        return isChanged(oldStrategy, newStrategy, null) ||
                isBidChangedWithLimitProblem(problem, oldStrategy, newStrategy);
    }

    private boolean isBidChangedWithLimitProblem(AutobudgetAggregatedHourlyProblem problem,
                                                 @Nullable DbStrategy oldStrategy, @Nullable DbStrategy newStrategy) {
        return (problem == MAX_BID_REACHED ||
                problem == ENGINE_MIN_COST_LIMITED ||
                problem == WALLET_DAY_BUDGET_REACHED) &&
                (getBid(oldStrategy) != null && getBid(newStrategy) != null
                        && getBid(oldStrategy).compareTo(getBid(newStrategy)) != 0);
    }

    //для разных стартегий - разное название поля со ставкой
    private static BigDecimal getBid(DbStrategy strategy) {
        if (strategy.getStrategyData().getBid() != null) {
            return strategy.getStrategyData().getBid();
        }
        if (strategy.getStrategyData().getAvgBid() != null) {
            return strategy.getStrategyData().getAvgBid();
        }
        return null;
    }

    private boolean hasAvgCpaChanged(@Nullable DbStrategy oldStrategy, @Nullable DbStrategy newStrategy) {
        return isChanged(oldStrategy, newStrategy, null) ||
                isChanged(oldStrategy, newStrategy, StrategyData.AVG_CPA);
    }

    public AlertsFreezeInfo getCampaignsToFreezeAlertsOnContextLimitChange(ClientId clientId,
                                                                           Collection<AppliedChanges<CampaignWithNetworkSettings>> campaigns
    ) {
        var contextLimitChanged = campaigns
                .stream()
                .filter(this::isContextLimitChanged)
                .map(ac -> ac.getModel().getId())
                .collect(Collectors.toSet());
        return getCampaignsToFreezeAlerts(clientId, contextLimitChanged);
    }

    public AlertsFreezeInfo getCampaignsToFreezeAlerts(ClientId clientId, Collection<Long> campaignIds) {
        var campaignsToFreeze = EntryStream.of(getAggregatedHourlyProblems(clientId, campaignIds))
                .filterValues(problem -> problem == MAX_BID_REACHED || problem == UPPER_POSITIONS_REACHED)
                .keys()
                .toSet();
        return new AlertsFreezeInfo(campaignsToFreeze);
    }

    public static boolean isContextLimitChanged(int oldContextLimit, int newContextLimit) {
        // включены показы на тематических площадках
        if (oldContextLimit == SHOWS_DISABLED_CONTEXT_LIMIT &&
                newContextLimit <= MAX_CONTEXT_LIMIT &&
                newContextLimit > AUTO_CONTEXT_LIMIT) {
            return true;
        }

        // увеличен % расхода на тематические площадки, но не морозим, если отлючаем РСЯ
        return oldContextLimit < newContextLimit && newContextLimit != SHOWS_DISABLED_CONTEXT_LIMIT;
    }

    private boolean isContextLimitChanged(AppliedChanges<CampaignWithNetworkSettings> ac) {
        //noinspection ConstantConditions
        int oldContextLimit = ac.getOldValue(CampaignWithNetworkSettings.CONTEXT_LIMIT);
        //noinspection ConstantConditions
        int newContextLimit = ac.getNewValue(CampaignWithNetworkSettings.CONTEXT_LIMIT);
        return isContextLimitChanged(oldContextLimit, newContextLimit);
    }
}
