package ru.yandex.direct.core.entity.autobudget.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.autobudget.model.CpaAutobudgetPessimizedUser;
import ru.yandex.direct.core.entity.autobudget.repository.CpaAutobudgetPessimizedUsersRepository;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class CpaAutobudgetPessimizedUsersService {

    private final CpaAutobudgetPessimizedUsersRepository cpaAutobudgetPessimizedUsersRepository;

    @Autowired
    public CpaAutobudgetPessimizedUsersService(CpaAutobudgetPessimizedUsersRepository cpaAutobudgetPessimizedUsersRepository) {
        this.cpaAutobudgetPessimizedUsersRepository = cpaAutobudgetPessimizedUsersRepository;
    }

    /**
     * Проверяет есть ли clientId в списке пессимизированных CPA-автобюджетом пользователей
     */
    public boolean isLoginPessimized(ClientId clientId) {
        return isClientIdsPessimized(Set.of(clientId)).getOrDefault(clientId, false);
    }

    public Map<ClientId, Boolean> isClientIdsPessimized(Collection<ClientId> clientIds) {
        List<Long> convertedClientIds = mapList(clientIds, ClientId::asLong);
        return StreamEx.of(cpaAutobudgetPessimizedUsersRepository.getPessimizedUsersByClientId(convertedClientIds))
                .mapToEntry(CpaAutobudgetPessimizedUser::getClientId, this::isPessimized)
                .mapKeys(ClientId::fromLong)
                .toMap();
    }

    private boolean isPessimized(CpaAutobudgetPessimizedUser user) {
        return user.getMinBidTargetCpaMultiplier()
                .compareTo(BigDecimal.ONE) < 0;
    }

    /**
     * Добавить или обновить данные о пессимизированных логинах
     */
    public int addOrUpdatePessimizedLogins(Collection<CpaAutobudgetPessimizedUser> logins) {
        return cpaAutobudgetPessimizedUsersRepository.addOrUpdatePessimizedUsers(logins);
    }

    /**
     * Удалить все пессимизированные логины старше заданного времени
     *
     * @param boundaryUpdateTime - пороговое время, записи старше которого удаляем
     */
    public int deleteLoginsWithUpdateTimeOlderThan(LocalDateTime boundaryUpdateTime) {
        return cpaAutobudgetPessimizedUsersRepository.deletePessimizedUsersOlderThan(boundaryUpdateTime);
    }
}
