package ru.yandex.direct.core.entity.autooverdraftmail.repository;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.autooverdraft.model.ClientOverdraftLimitChanges;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.tables.ClientOverdraftLimitChanges.CLIENT_OVERDRAFT_LIMIT_CHANGES;
import static ru.yandex.direct.dbschema.ppc.tables.Clients.CLIENTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;

@Repository
@ParametersAreNonnullByDefault
public class ClientOverdraftLimitChangesRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<ClientOverdraftLimitChanges> mapper;

    @Autowired
    public ClientOverdraftLimitChangesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = JooqMapperWithSupplierBuilder.builder(ClientOverdraftLimitChanges::new)
                .map(convertibleProperty(ClientOverdraftLimitChanges.CLIENT_ID,
                        CLIENT_OVERDRAFT_LIMIT_CHANGES.CLIENT_ID,
                        ClientId::fromLong,
                        ClientId::asLong))
                .map(convertibleProperty(ClientOverdraftLimitChanges.IS_NOTIFICATION_SENT,
                        CLIENT_OVERDRAFT_LIMIT_CHANGES.IS_NOTIFICATION_SENT,
                        RepositoryUtils::booleanFromLong,
                        RepositoryUtils::booleanToLong
                ))
                .build();
    }

    /**
     * Добавить запись, если запись есть, то заменить её
     *
     * @param shard — шард
     * @param clientId — id клиента для записи
     * @param isSent — отправлено ли письмо (если false, то не отправлено)
     */
    public void add(int shard, ClientId clientId, boolean isSent) {
        new InsertHelper<>(dslContextProvider.ppc(shard), CLIENT_OVERDRAFT_LIMIT_CHANGES)
                .add(mapper, new ClientOverdraftLimitChanges()
                        .withClientId(clientId).withIsNotificationSent(isSent))
                    .onDuplicateKeyUpdate()
                    .set(CLIENT_OVERDRAFT_LIMIT_CHANGES.CLIENT_ID, clientId.asLong())
                    .set(CLIENT_OVERDRAFT_LIMIT_CHANGES.IS_NOTIFICATION_SENT, RepositoryUtils.booleanToLong(isSent))
                .execute();
    }

    /**
     * Удалить записи с переданными id клиентов
     *
     * @param shard — шард
     * @param clientIds — список id клиентов
     * @return количество удалённых записей
     */
    public int deleteByClientIds(int shard, List<ClientId> clientIds) {
        List<Long> clientIdsAsLongs = clientIds.stream().map(ClientId::asLong).collect(Collectors.toList());
        return dslContextProvider.ppc(shard)
                .deleteFrom(CLIENT_OVERDRAFT_LIMIT_CHANGES)
                .where(CLIENT_OVERDRAFT_LIMIT_CHANGES.CLIENT_ID.in(clientIdsAsLongs))
                .execute();
    }

    /**
     * Получить все записи
     *
     * @param shard — шард
     * @return все записи в шарде
     */
    @QueryWithoutIndex("SELECT без индекса — это прямое назначение таблицы")
    public List<ClientOverdraftLimitChanges> getAll(int shard) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CLIENT_OVERDRAFT_LIMIT_CHANGES)
                // берём большую таблицу во вторую очередь
                .straightJoin(CLIENTS)
                    // защита от разломанного шардинга
                    .on(CLIENT_OVERDRAFT_LIMIT_CHANGES.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .fetch(mapper::fromDb);
    }
}
