package ru.yandex.direct.core.entity.autoprice.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.enums.AutoPriceCampQueueStatus;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.dbschema.ppc.tables.AutoPriceCampQueue.AUTO_PRICE_CAMP_QUEUE;

/**
 * Работа с очередью оффлайн конструктора цен
 */
@ParametersAreNonnullByDefault
@Repository
public class AutoPriceCampQueueRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public AutoPriceCampQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить список id заданий, добавленных до {@code borderDate} и в состоянии не Wait
     *
     * @param shard      шард
     * @param borderDate {@link LocalDateTime} граничное время
     * @return {@link List} id заданий
     */
    public List<Long> getOutdatedIds(int shard, LocalDateTime borderDate) {
        return dslContextProvider.ppc(shard)
                .select(AUTO_PRICE_CAMP_QUEUE.ID)
                .from(AUTO_PRICE_CAMP_QUEUE)
                .where(AUTO_PRICE_CAMP_QUEUE.STATUS.ne(AutoPriceCampQueueStatus.Wait))
                .and(AUTO_PRICE_CAMP_QUEUE.ADD_TIME.lt(borderDate))
                .fetch(AUTO_PRICE_CAMP_QUEUE.ID);
    }

    /**
     * Получение идентификаторов записей для списка кампаний и статусов
     */
    public List<Long> getQueueIds(int shard, Collection<Long> campaignIds, List<AutoPriceCampQueueStatus> statuses) {
        if (campaignIds.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(AUTO_PRICE_CAMP_QUEUE.ID)
                .from(AUTO_PRICE_CAMP_QUEUE)
                .where(AUTO_PRICE_CAMP_QUEUE.STATUS.in(statuses))
                .and(AUTO_PRICE_CAMP_QUEUE.CID.in(campaignIds))
                .fetch(AUTO_PRICE_CAMP_QUEUE.ID);
    }

    /**
     * Выставляет статус {@link AutoPriceCampQueueStatus#Error} ошибкой {@code error} для переданных {@code recordIds}
     */
    public void setErrorStatus(int shard, List<Long> recordIds, String error) {
        if (recordIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard)
                .update(AUTO_PRICE_CAMP_QUEUE)
                .set(AUTO_PRICE_CAMP_QUEUE.STATUS, AutoPriceCampQueueStatus.Error)
                .set(AUTO_PRICE_CAMP_QUEUE.ERROR_STR, error)
                .set(AUTO_PRICE_CAMP_QUEUE.SEND_TIME, LocalDateTime.now())
                .where(AUTO_PRICE_CAMP_QUEUE.ID.in(recordIds))
                .execute();
    }

    /**
     * Удаляет задания из очереди по их id
     *
     * @param shard шард
     * @param ids   {@link Collection} id заданий для удаления
     * @return число удаленных записей
     */
    public int deleteByIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(AUTO_PRICE_CAMP_QUEUE)
                .where(AUTO_PRICE_CAMP_QUEUE.ID.in(ids))
                .execute();
    }

}
