package ru.yandex.direct.core.entity.balance.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueItem;
import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueObjType;
import ru.yandex.direct.core.entity.balance.model.BalanceNotificationInfo;
import ru.yandex.direct.dbschema.ppc.enums.BalanceInfoQueueSendStatus;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static org.jooq.impl.DSL.or;
import static ru.yandex.direct.dbschema.ppc.tables.BalanceInfoQueue.BALANCE_INFO_QUEUE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

/**
 * Работа с очередью переотправки в баланс (ppc.balance_info_queue)
 */
@ParametersAreNonnullByDefault
@Repository
public class BalanceInfoQueueRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<BalanceInfoQueueItem> jooqMapper;

    @Autowired
    public BalanceInfoQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqMapper = JooqMapperWithSupplierBuilder.builder(BalanceInfoQueueItem::new)
                .map(property(BalanceInfoQueueItem.CID_OR_UID, BALANCE_INFO_QUEUE.CID_OR_UID))
                .map(convertibleProperty(BalanceInfoQueueItem.OBJ_TYPE, BALANCE_INFO_QUEUE.OBJ_TYPE,
                        BalanceInfoQueueObjType::fromSource, BalanceInfoQueueObjType::toSource))
                .map(property(BalanceInfoQueueItem.OPERATOR_UID, BALANCE_INFO_QUEUE.OPERATOR_UID))
                .map(convertibleProperty(BalanceInfoQueueItem.SEND_STATUS, BALANCE_INFO_QUEUE.SEND_STATUS,
                        ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueSendStatus::fromSource,
                        ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueSendStatus::toSource))
                .readProperty(BalanceInfoQueueItem.ADD_TIME, fromField(BALANCE_INFO_QUEUE.ADD_TIME))
                .map(property(BalanceInfoQueueItem.PRIORITY, BALANCE_INFO_QUEUE.PRIORITY))
                .build();
    }

    /**
     * Возвращает id записей подходящих для удаления:
     * отправка завершилась ошибкой (send_status = Error) и дата добавления раньше boundaryDate
     * или
     * отправка прошла успешно (send_status = Send)
     *
     * @param shard        шард
     * @param boundaryDate {@link LocalDateTime} граничная дата
     * @param limitOffset  предельное кол-во записей на чтение и смещение с начала выборки
     * @return список id удовлетворяющих условию
     */
    @QueryWithoutIndex("Используется только в jobs")
    public List<Long> getSentOrOutdatedErrorIds(int shard, LocalDateTime boundaryDate, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(BALANCE_INFO_QUEUE.ID)
                .from(BALANCE_INFO_QUEUE)
                .where(or(BALANCE_INFO_QUEUE.SEND_STATUS.eq(BalanceInfoQueueSendStatus.Send),
                        BALANCE_INFO_QUEUE.SEND_STATUS.eq(BalanceInfoQueueSendStatus.Error)
                                .and(BALANCE_INFO_QUEUE.ADD_TIME.lt(boundaryDate))))
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(BALANCE_INFO_QUEUE.ID);
    }

    /**
     * Возвращает limit последних записей в очереди:
     *
     * @param shard       шард
     * @param limitOffset предельное кол-во записей на чтение и смещение с начала выборки
     * @return список последних записей в очереди
     */
    public List<BalanceInfoQueueItem> getLatestRecords(int shard, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(BALANCE_INFO_QUEUE)
                .orderBy(BALANCE_INFO_QUEUE.ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Удаление записей с заданными ids
     *
     * @param shard шард
     * @param ids   {@link Collection} id записей для удаления
     * @return количество удаленных записей
     * @
     */
    public int deleteByIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(BALANCE_INFO_QUEUE)
                .where(BALANCE_INFO_QUEUE.ID.in(ids))
                .execute();
    }

    /**
     * Возвращает записи из очереди в статусе Wait
     * записи так же фильтрируются по (cid_or_uid and obj_type)
     */
    public List<BalanceInfoQueueItem> getExistingRecordsInWaitStatus(DSLContext dslContext,
                                                                     Collection<BalanceNotificationInfo> notificationInfos) {
        if (notificationInfos.isEmpty()) {
            return Collections.emptyList();
        }

        Condition conditions = notificationInfos.stream()
                .map(ni -> BALANCE_INFO_QUEUE.CID_OR_UID.eq(ni.getCidOrUid())
                        .and(BALANCE_INFO_QUEUE.OBJ_TYPE.eq(BalanceInfoQueueObjType.toSource(ni.getObjType()))))
                .reduce(DSL.noCondition(), Condition::or);

        return dslContext
                .select(jooqMapper.getFieldsToRead())
                .from(BALANCE_INFO_QUEUE)
                .where(conditions
                        .and(BALANCE_INFO_QUEUE.SEND_STATUS.eq(BalanceInfoQueueSendStatus.Wait)))
                .fetch(jooqMapper::fromDb);
    }

    public int addToBalanceInfoQueue(int shard, Collection<BalanceInfoQueueItem> itemsToAdd) {
        return addToBalanceInfoQueue(dslContextProvider.ppc(shard), itemsToAdd);
    }

    public int addToBalanceInfoQueue(DSLContext dslContext, Collection<BalanceInfoQueueItem> itemsToAdd) {
        return new InsertHelper<>(dslContext, BALANCE_INFO_QUEUE)
                .addAll(jooqMapper, itemsToAdd)
                .executeIfRecordsAdded();
    }

}
