package ru.yandex.direct.core.entity.balance.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueItem;
import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueObjType;
import ru.yandex.direct.core.entity.balance.model.BalanceInfoQueueSendStatus;
import ru.yandex.direct.core.entity.balance.model.BalanceNotificationInfo;
import ru.yandex.direct.core.entity.balance.repository.BalanceInfoQueueRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;

@Service
@ParametersAreNonnullByDefault
public class BalanceInfoQueueService {

    private static final Logger logger = LoggerFactory.getLogger(BalanceInfoQueueService.class);

    private final BalanceInfoQueueRepository balanceInfoQueueRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public BalanceInfoQueueService(BalanceInfoQueueRepository balanceInfoQueueRepository,
                                   ShardHelper shardHelper) {
        this.balanceInfoQueueRepository = balanceInfoQueueRepository;
        this.shardHelper = shardHelper;
    }


    /**
     * Возвращает limit последних записей в очереди:
     *
     * @param limit предельное кол-во записей на чтение
     * @return список последних записей в очереди
     */
    public List<BalanceInfoQueueItem> getLatestRecords(int limit) {
        List<BalanceInfoQueueItem> result = new ArrayList<>();
        shardHelper.forEachShard(
                shard -> result.addAll(balanceInfoQueueRepository.getLatestRecords(shard, limited(limit))));
        return result;
    }

    /**
     * Добавить нотификации в очередь ленивой переотправки в Баланс
     */
    //TODO добавить тесты DIRECT-103676
    public void addToBalanceInfoQueue(DSLContext dslContext, Collection<BalanceNotificationInfo> notificationInfos) {
        /* TODO из перла ): после отказа от хранения обработанных и ошибочных (от send_status) завести уникальный ключ
            obj_type+cid_or_uid и вставлять одним запросом с ON DUPLICATE KEY UPDATE, исключая текущие два запроса и
            гонки
         */
        List<BalanceInfoQueueItem> existingItemsInQueue = balanceInfoQueueRepository
                .getExistingRecordsInWaitStatus(dslContext, notificationInfos);

        List<BalanceInfoQueueItem> itemsToAdd = getBalanceInfoQueueItemsToAdd(existingItemsInQueue, notificationInfos);
        int addedItemsCount = balanceInfoQueueRepository.addToBalanceInfoQueue(dslContext, itemsToAdd);
        logger.debug("{} balance notifications added", addedItemsCount);
    }

    private static List<BalanceInfoQueueItem> getBalanceInfoQueueItemsToAdd(
            Collection<BalanceInfoQueueItem> existingBalanceInfoQueueItems,
            Collection<BalanceNotificationInfo> notificationInfos) {
        Map<Pair<BalanceInfoQueueObjType, Long>, Long> priorityByObjTypeAndCidOrUid =
                StreamEx.of(existingBalanceInfoQueueItems)
                        .mapToEntry(item -> Pair.of(item.getObjType(), item.getCidOrUid()),
                                BalanceInfoQueueItem::getPriority)
                        .toMap(Long::max);

        /*
        если объект уже стоит в очереди с большим или таким же приоритетом, заново не ставим.
        с меньшим - ставим второй раз.
         */
        return filterAndMapList(notificationInfos,
                ni -> ni.getPriority().getTypedValue() > priorityByObjTypeAndCidOrUid
                        .getOrDefault(Pair.of(ni.getObjType(), ni.getCidOrUid()), Long.MIN_VALUE),
                BalanceInfoQueueService::toBalanceInfoQueueItems);
    }

    private static BalanceInfoQueueItem toBalanceInfoQueueItems(BalanceNotificationInfo notificationInfo) {
        return new BalanceInfoQueueItem()
                .withCidOrUid(notificationInfo.getCidOrUid())
                .withObjType(notificationInfo.getObjType())
                .withPriority(notificationInfo.getPriority().getTypedValue().longValue())
                .withOperatorUid(notificationInfo.getOperatorUid())
                .withSendStatus(BalanceInfoQueueSendStatus.WAIT);
    }

}
