package ru.yandex.direct.core.entity.balanceaggrmigration.lock;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.redislock.DistributedLockException;
import ru.yandex.direct.redislock.lettuce.LettuceLockBuilder;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;

@ParametersAreNonnullByDefault
@Service
public class AggregateMigrationRedisLockService {
    private static final Logger logger = LoggerFactory.getLogger(AggregateMigrationRedisLockService.class);

    private static final String MIGRATION_BALANCE_CONFIG_BRANCH = "migration-balance-lock";

    private final DirectConfig lockConfig;
    private final LettuceConnectionProvider lettuce;

    public AggregateMigrationRedisLockService(DirectConfig directConfig,
                                              @Qualifier(LETTUCE) LettuceConnectionProvider lettuce) {
        this.lockConfig = directConfig.getBranch(MIGRATION_BALANCE_CONFIG_BRANCH);
        this.lettuce = lettuce;
    }

    public DistributedLock buildLock(long walletId) {
        int maxLocks = lockConfig.getInt("max-locks");
        long negotiationTimeout = lockConfig.getInt("negotiation-timeout");
        String lockKeyPrefix = lockConfig.getString("lock-key-prefix");
        String lockKey = "group" + (walletId % 3);

        LettuceLockBuilder lockBuilder = LettuceLockBuilder.newBuilder(lettuce::getConnection)
                .withMaxLocks(maxLocks)
                .withTTL(negotiationTimeout)
                .withKeyPrefix(lockKeyPrefix + "-" + String.valueOf(walletId));

        return lockBuilder.createLock(lockKey);
    }

    public DistributedLock lock(long walletId) {
        DistributedLock lock = buildLock(walletId);

        try {
            lock.lock();
        } catch (DistributedLockException | InterruptedException e) {
            logger.error("[LOCK EXCEPTION] cant acquire migration lock: lock service exception", e);
        }

        return lock;
    }

    public boolean unlock(DistributedLock lock) {
        boolean isUnlocked = false;

        try {
            lock.unlockByEntry();
            isUnlocked = true;
        } catch (DistributedLockException e) {
            logger.error("[UNLOCK] cant release migration lock: unlock migration exception", e);
        }

        return isUnlocked;
    }

}
