package ru.yandex.direct.core.entity.banner.container;

import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Контейнер необходимый для валидации пикселей
 */
@ParametersAreNonnullByDefault
public class BannerWithPixelsValidationContainerImpl implements BannerWithPixelsValidationContainer {

    private final int shard;
    private final ClientId clientId;

    private final Set<String> clientEnabledFeatures;
    private final Map<Integer, AdGroupForBannerOperation> indexToAdGroupMap;
    private final IdentityHashMap<Banner, Integer> bannerToIndexMap;

    private final Map<Integer, CommonCampaign> indexToCampaignMap;

    /**
     * Индекс добавляемого баннера -> информация о нём из комплексной операции.
     * Используется для валидации баннеров в рамках комплексной операции.
     */
    private final Map<Integer, BannerValidationInfo> bannersValidationInfoMap;

    public BannerWithPixelsValidationContainerImpl(int shard,
                                                   ClientId clientId,
                                                   Set<String> clientEnabledFeatures,
                                                   Map<Integer, AdGroupForBannerOperation> indexToAdGroupMap,
                                                   IdentityHashMap<Banner, Integer> bannerToIndexMap,
                                                   Map<Integer, CommonCampaign> indexToCampaignMap,
                                                   Map<Integer, BannerValidationInfo> bannersValidationInfoMap) {
        this.shard = shard;
        this.clientId = clientId;
        this.clientEnabledFeatures = clientEnabledFeatures;
        this.indexToAdGroupMap = indexToAdGroupMap;
        this.bannerToIndexMap = bannerToIndexMap;
        this.indexToCampaignMap = indexToCampaignMap;
        this.bannersValidationInfoMap = bannersValidationInfoMap;
    }

    @Override
    public int getShard() {
        return shard;
    }

    @Override
    public ClientId getClientId() {
        return clientId;
    }

    @Override
    public Collection<CommonCampaign> getCampaigns() {
        checkIndexToCampaignMapIsSet();
        return indexToCampaignMap.values();
    }

    @Override
    public Collection<AdGroupForBannerOperation> getAdGroups() {
        checkIndexToAdGroupMapIsSet();
        return indexToAdGroupMap.values();
    }

    /**
     * Информация о группе.
     * Может вернуть null в следующем случае:
     * При добавлении баннера указанный id группы ссылается на группу, отсутствующую у клиента
     */
    @Nullable
    @Override
    public AdGroupForBannerOperation getAdGroup(Banner banner) {
        checkIndexToAdGroupMapIsSet();
        Integer bannerIndex = getBannerIndex(banner);
        return indexToAdGroupMap.get(bannerIndex);
    }

    /**
     * Получить список уникальных групп.
     * В {@link #getAdGroups()} могут храниться дубли групп.
     */
    @Override
    public Collection<AdGroupForBannerOperation> getUniqueAdGroups() {
        checkIndexToAdGroupMapIsSet();
        return StreamEx.of(indexToAdGroupMap.values())
                .distinct(AdGroupForBannerOperation::getId)
                .toList();
    }

    @Nullable
    @Override
    public BannerValidationInfo getBannersValidationInfo(Banner banner) {
        Integer bannerIndex = getBannerIndex(banner);
        return ifNotNull(bannersValidationInfoMap, map -> map.get(bannerIndex));
    }

    @Override
    public Set<String> getClientEnabledFeatures() {
        return clientEnabledFeatures;
    }

    @Nullable
    private Integer getBannerIndex(Banner banner) {
        checkNotNull(bannerToIndexMap, "bannerToIndexMap is not set yet");
        return bannerToIndexMap.get(banner);
    }

    /**
     * Индекс добавляемого баннера -> информация о нём из комплексной операции.
     */
    @Nullable
    @Override
    public Map<Integer, BannerValidationInfo> getBannersValidationInfoMap() {
        return bannersValidationInfoMap;
    }

    @Override
    public CommonCampaign getCampaign(Banner banner) {
        checkIndexToCampaignMapIsSet();
        Integer bannerIndex = getBannerIndex(banner);
        return indexToCampaignMap.get(bannerIndex);
    }

    private void checkIndexToCampaignMapIsSet() {
        checkNotNull(indexToCampaignMap, "indexToCampaignMap is not set yet");
    }

    private void checkIndexToAdGroupMapIsSet() {
        checkState(indexToAdGroupMap != null, "indexToAdGroupForOperationMap is not set yet");
    }

    @Override
    public boolean isFeatureEnabledForClient(FeatureName featureName) {
        return getClientEnabledFeatures().contains(featureName.getName());
    }
}
