package ru.yandex.direct.core.entity.banner.container;

import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganizationAndPhone;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeHelper;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneIdsByTypeContainer;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.CommonUtils;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.banner.container.BannerOperationContainerUtils.needToCheckPermalinkAccessForAdd;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getValidPermalinkIds;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

@Service
public class BannersAddOperationContainerService extends BannersOperationContainerService {

    private final AdGroupRepository adGroupRepository;
    private final VcardRepository vcardRepository;

    @Autowired
    public BannersAddOperationContainerService(CampaignTypedRepository campaignTypedRepository,
                                               PpcPropertiesSupport ppcPropertiesSupport,
                                               BannerWithCreativeHelper bannerWithCreativeHelper,
                                               CreativeService creativeService,
                                               AdGroupRepository adGroupRepository,
                                               OrganizationService organizationService,
                                               BannerImageRepository bannerImageRepository,
                                               BannerImageFormatRepository bannerImageFormatRepository,
                                               SitelinkSetRepository sitelinkSetRepository,
                                               TurboLandingRepository turboLandingRepository,
                                               PricePackageRepository pricePackageRepository,
                                               VcardRepository vcardRepository,
                                               ClientPhoneRepository clientPhoneRepository,
                                               MobileAppRepository mobileAppRepository,
                                               NetAcl netAcl) {
        super(campaignTypedRepository, ppcPropertiesSupport, bannerWithCreativeHelper, creativeService,
                sitelinkSetRepository, turboLandingRepository, pricePackageRepository, bannerImageRepository, bannerImageFormatRepository,
                clientPhoneRepository, mobileAppRepository, organizationService, netAcl);

        this.adGroupRepository = adGroupRepository;
        this.vcardRepository = vcardRepository;
    }

    public void fillContainers(BannersAddOperationContainerImpl container, List<BannerWithAdGroupId> banners) {
        IdentityHashMap<Banner, Integer> bannerToIndexMap = EntryStream.of(banners)
                .mapValues(b -> (Banner) b)
                .invert()
                .toCustomMap(IdentityHashMap::new);
        container.setBannerToIndexMap(bannerToIndexMap);

        Set<Long> adGroupIds = banners.stream()
                .map(BannerWithAdGroupId::getAdGroupId)
                .filter(CommonUtils::isValidId)
                .collect(Collectors.toSet());

        if (!container.isIndexToAdGroupMapSet()) {
            Map<Integer, AdGroupForBannerOperation> adGroupsMap = getIndexToAdGroupMap(container, banners, adGroupIds);
            initializeAdGroupsAndCampaignMapsInContainers(container, adGroupsMap);
        }

        if (!container.isIndexToContentPromotionAdgroupTypeSupplierSet()) {
            Map<Integer, ContentPromotionAdgroupType> contentPromotionAdgroupTypeMap =
                    getIndexToContentPromotionAdGroupTypeMap(container, banners, adGroupIds);
            container.setIndexToContentPromotionAdgroupTypeMap(() -> contentPromotionAdgroupTypeMap);
        }

        fillContainerPpcProperties(container);
        fillCreativeMap(container, banners);
        fillSitelinkSets(container, banners);
        fillVcards(container, banners);
        fillTurboLandings(container, banners, container.getSitelinkSets());

        var bannersWithOrganization = selectList(banners, BannerWithOrganization.class);

        Map<Long, Organization> clientOrganizations = getClientOrganizations(
                container.getClientId(),
                bannersWithOrganization);
        container.setClientOrganizations(clientOrganizations);
        var bannersWithOrganizationAndPhone = selectList(bannersWithOrganization, BannerWithOrganizationAndPhone.class);
        fillOrganizationPhones(container, bannersWithOrganizationAndPhone);
        fillImageFormats(container, banners);
    }

    public <T extends Banner> void fillImageFormats(BannersAddOperationContainerImpl container,
                                                    List<T> banners) {
        fillExistingImageTypesWithSizes(container, banners);
        fillExistingBigKingImageHashesWithType(container, banners);
        fillImages(container, banners);
    }

    public void fillOrganizationPhones(BannersAddOperationContainerImpl container,
                                       List<BannerWithOrganizationAndPhone> banners) {
        var permalinkIds = getValidPermalinkIds(banners);
        var phoneIdsContainer = fillAllowedPhoneIds(container, permalinkIds);
        phoneIdsContainer.setAllowOverrideNotManualPhoneWithNoRights(container, netAcl);
        boolean allowOverrideNotManual = phoneIdsContainer.isAllowedOverrideNotManual();
        Set<Long> permalinksToAccessCheck = container.isCopy() ? emptySet() :
                extractPermalinksToAccessCheckForAdd(banners, phoneIdsContainer, container,
                        allowOverrideNotManual);
        fillContainerAccessibleOrganizationsPermalinkId(container, permalinksToAccessCheck);
    }

    /**
     * Возвращает множество пермалинков, у которых нужно будет проверить доступ при добавлении баннера.
     * Доступ нужно проверить, если:
     * <ul>
     * <li>при выключенных фичах {@link ru.yandex.direct.feature.FeatureName#TELEPHONY_ALLOWED}
     * и {@link ru.yandex.direct.feature.FeatureName#UNIVERSAL_CAMPAIGNS_ENABLED_FOR_UAC}
     * если добавляется любой номер
     * </li>
     * <li>
     * при хотя бы одной включенной фиче {@link ru.yandex.direct.feature.FeatureName#TELEPHONY_ALLOWED}
     * или {@link ru.yandex.direct.feature.FeatureName#UNIVERSAL_CAMPAIGNS_ENABLED_FOR_UAC}
     * если добавляется ручной номер
     * </li>
     * </ul>
     */
    public static Set<Long> extractPermalinksToAccessCheckForAdd(
            Collection<BannerWithOrganizationAndPhone> banners,
            ClientPhoneIdsByTypeContainer phoneIdsContainer,
            BannersAddOperationContainer bannersContainer,
            boolean allowOverrideNotManual
    ) {
        return StreamEx.of(banners)
                .filter(b -> needToCheckPermalinkAccessForAdd(b,
                        phoneIdsContainer, bannersContainer, allowOverrideNotManual))
                .map(BannerWithOrganizationAndPhone::getPermalinkId)
                .filter(CommonUtils::isValidId)
                .toSet();
    }

    private Map<Long, Organization> getClientOrganizations(
            ClientId clientId,
            Collection<BannerWithOrganization> banners) {
        List<Long> permalinkIds = getValidPermalinkIds(banners);
        return organizationService.getClientOrganizations(permalinkIds, clientId);
    }

    /**
     * Извлекаем информацию о типах контента в контент-промоушн-группах из базы
     *
     * @return позиция баннера -> ContentPromotionAdgroupType
     */
    private Map<Integer, ContentPromotionAdgroupType> getIndexToContentPromotionAdGroupTypeMap(
            AbstractBannersOperationContainer container,
            List<BannerWithAdGroupId> banners,
            Set<Long> adGroupIds) {
        Map<Long, ContentPromotionAdgroupType> adGroupIdToContentPromotionType =
                adGroupRepository.getContentPromotionAdGroupTypesByIds(
                        container.getShard(), container.getClientId(), adGroupIds);
        return EntryStream.of(banners)
                .mapValues(BannerWithAdGroupId::getAdGroupId)
                .nonNullValues()
                .mapValues(adGroupIdToContentPromotionType::get)
                .nonNullValues()
                .toMap();
    }

    public <T extends Banner> void fillVcards(AbstractBannersOperationContainer container, List<T> banners) {
        Set<Long> vcardIds = StreamEx.of(banners)
                .nonNull()
                .select(BannerWithVcard.class)
                .map(BannerWithVcard::getVcardId)
                .nonNull()
                .toSet();

        var vcards = vcardRepository.getVcards(container.getShard(), container.getChiefUid(), vcardIds);
        container.setVcardIdToData(listToMap(vcards, Vcard::getId));
    }

    /**
     * Извлекаем информацию о группах из базы
     *
     * @return позиция баннера -> AdGroupForBannerOperation
     */
    private Map<Integer, AdGroupForBannerOperation> getIndexToAdGroupMap(AbstractBannersOperationContainer container,
                                                                         List<BannerWithAdGroupId> banners,
                                                                         Set<Long> adGroupIds) {
        Map<Long, AdGroupForBannerOperation> adGroupsInfo =
                adGroupRepository.getAdGroupsForBannerOperation(container.getShard(), adGroupIds,
                        container.getClientId());

        return EntryStream.of(banners)
                .mapValues(banner -> adGroupsInfo.get(banner.getAdGroupId()))
                .nonNullValues()
                .toMap();
    }

}
