package ru.yandex.direct.core.entity.banner.container;

import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithBigKingImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeHelper;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithContentLanguage;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMobileContent;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneIdsByTypeContainer;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.db.PpcPropertiesSupport.deserialize;
import static ru.yandex.direct.common.db.PpcPropertyNames.CPM_GEOPRODUCT_AUTO_MODERATION;
import static ru.yandex.direct.common.db.PpcPropertyNames.CPM_GEO_PIN_AUTO_MODERATION;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIdsFilter;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

/**
 * Сервис, который помогает заполнять контейнер операций BannersAddOperation и BannersUpdateOperation
 */
@Service
public class BannersOperationContainerService {

    private final CampaignTypedRepository campaignTypedRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final BannerWithCreativeHelper bannerWithCreativeHelper;
    private final CreativeService creativeService;
    private final SitelinkSetRepository sitelinkSetRepository;
    private final TurboLandingRepository turboLandingRepository;
    private final PricePackageRepository pricePackageRepository;
    private final BannerImageRepository bannerImageRepository;
    protected final BannerImageFormatRepository bannerImageFormatRepository;
    protected final ClientPhoneRepository clientPhoneRepository;
    protected final MobileAppRepository mobileAppRepository;
    protected final OrganizationService organizationService;
    protected final NetAcl netAcl;

    @Autowired
    public BannersOperationContainerService(CampaignTypedRepository campaignTypedRepository,
                                            PpcPropertiesSupport ppcPropertiesSupport,
                                            BannerWithCreativeHelper bannerWithCreativeHelper,
                                            CreativeService creativeService,
                                            SitelinkSetRepository sitelinkSetRepository,
                                            TurboLandingRepository turboLandingRepository,
                                            PricePackageRepository pricePackageRepository,
                                            BannerImageRepository bannerImageRepository,
                                            BannerImageFormatRepository bannerImageFormatRepository,
                                            ClientPhoneRepository clientPhoneRepository,
                                            MobileAppRepository mobileAppRepository,
                                            OrganizationService organizationService,
                                            NetAcl netAcl) {
        this.campaignTypedRepository = campaignTypedRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.bannerWithCreativeHelper = bannerWithCreativeHelper;
        this.creativeService = creativeService;
        this.sitelinkSetRepository = sitelinkSetRepository;
        this.turboLandingRepository = turboLandingRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.bannerImageRepository = bannerImageRepository;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.clientPhoneRepository = clientPhoneRepository;
        this.mobileAppRepository = mobileAppRepository;
        this.organizationService = organizationService;
        this.netAcl = netAcl;
    }

    public void initializeAdGroupsAndCampaignMapsInContainers(
            AbstractBannersOperationContainer container,
            Map<Integer, AdGroupForBannerOperation> indexToAdGroupMap) {
        container.setIndexToAdGroupMap(indexToAdGroupMap);
        Set<Long> campaignIds = indexToAdGroupMap.values().stream()
                .map(AdGroupForBannerOperation::getCampaignId)
                .collect(toSet());

        fillContainerCampaignsMaps(container, indexToAdGroupMap, campaignIds);
        fillMobileAppTrackers(container);
    }

    private void fillContainerCampaignsMaps(
            AbstractBannersOperationContainer container,
            Map<Integer, AdGroupForBannerOperation> indexToAdgroup,
            Set<Long> campaignIds) {
        List<BaseCampaign> baseCampaigns = campaignTypedRepository.getSafely(container.getShard(),
                campaignIdsFilter(campaignIds),
                List.of(CommonCampaign.class, CampaignWithStrategy.class, CampaignWithContentLanguage.class,
                        CampaignWithMobileContent.class, CampaignWithPricePackage.class));

        Map<Long, CommonCampaign> campaigns = StreamEx.of(baseCampaigns)
                .select(CommonCampaign.class)
                .toMap(CommonCampaign::getId, identity());

        Map<Integer, CommonCampaign> indexToCampaignMap = EntryStream.of(indexToAdgroup)
                .mapValues(adGroup -> campaigns.get(adGroup.getCampaignId()))
                .nonNullValues()
                .toMap();

        container.setIndexToCampaignMap(indexToCampaignMap);

        Map<Long, CampaignWithStrategy> campaignIdToCampaignWithStrategyMap = StreamEx.of(baseCampaigns)
                .select(CampaignWithStrategy.class)
                .toMap(CampaignWithStrategy::getId, identity());
        container.setCampaignIdToCampaignWithStrategyMap(campaignIdToCampaignWithStrategyMap);

        Map<Long, CampaignWithContentLanguage> campaignIdToCampaignWithContentLanguageMap = StreamEx.of(baseCampaigns)
                .select(CampaignWithContentLanguage.class)
                .toMap(CampaignWithContentLanguage::getId, identity());
        container.setCampaignIdToCampaignWithContentLanguageMap(campaignIdToCampaignWithContentLanguageMap);

        Map<Long, CampaignWithMobileContent> campaignIdToCampaignWithMobileContentMap = StreamEx.of(baseCampaigns)
                .select(CampaignWithMobileContent.class)
                .toMap(CampaignWithMobileContent::getId, identity());
        container.setCampaignIdToCampaignWithMobileContentMap(campaignIdToCampaignWithMobileContentMap);

        Map<Long, Long> campaignIdToPricePackageIdMap = StreamEx.of(baseCampaigns)
                .select(CampaignWithPricePackage.class)
                .toMap(CampaignWithPricePackage::getId, CampaignWithPricePackage::getPricePackageId);
        if (!campaignIdToPricePackageIdMap.isEmpty()) {
            var pricePackages = pricePackageRepository.getPricePackages(campaignIdToPricePackageIdMap.values());
            container.setCampaignIdToPricePackageMap(StreamEx.of(campaignIdToPricePackageIdMap.keySet())
                    .toMap(identity(), cid -> pricePackages.get(campaignIdToPricePackageIdMap.get(cid))));
        }
    }

    protected void fillContainerAccessibleOrganizationsPermalinkId(AbstractBannersOperationContainer container,
                                                                   Collection<Long> permalinkIds) {
        if (permalinkIds.isEmpty()) {
            container.setAccessibleOrganizationsPermalinkId(Set.of());
            return;
        }

        Map<Long, Boolean> permalinkIdToAccess = organizationService.hasAccess(container.getClientId(), permalinkIds);

        Set<Long> accessibleOrganizationsPermalinkId = EntryStream.of(permalinkIdToAccess)
                .filterValues(a -> a)
                .keys()
                .toSet();
        container.setAccessibleOrganizationsPermalinkId(accessibleOrganizationsPermalinkId);
    }

    /**
     * Заполняет контейнер списком идентификаторов номеров, которые принадлежат данным организациям
     * или имеют тип {@link ru.yandex.direct.core.entity.trackingphone.model.ClientPhoneType#MANUAL}
     */
    protected ClientPhoneIdsByTypeContainer fillAllowedPhoneIds(AbstractBannersOperationContainer container,
                                                                Collection<Long> permalinkIds) {
        if (permalinkIds.isEmpty()) {
            var phonesContainer = ClientPhoneIdsByTypeContainer.empty();
            container.setAllowedPhoneIds(phonesContainer);
            return phonesContainer;
        }
        var phonesContainer = new ClientPhoneIdsByTypeContainer();
        List<ClientPhone> clientPhones = clientPhoneRepository.getAllClientPhones(container.getClientId(),
                permalinkIds);
        clientPhones.forEach(phone -> {
            if (phone.getPermalinkId() == null) {
                phonesContainer.addManual(phone.getId());
            } else {
                phonesContainer.addNotManual(phone.getPermalinkId(), phone.getId());
            }
        });
        container.setAllowedPhoneIds(phonesContainer);
        return phonesContainer;
    }

    protected <T extends Banner> void fillCreativeMap(AbstractBannersOperationContainer container,
                                                      List<T> banners) {
        List<BannerWithCreative> bannerWithCreatives = selectList(banners, BannerWithCreative.class);
        Set<Long> creativeIds = bannerWithCreatives.stream()
                .map(BannerWithCreative::getCreativeId)
                .filter(Objects::nonNull)
                .collect(toSet());
        List<Creative> creatives = creativeService.get(container.getClientId(), creativeIds, null);
        Map<Long, Creative> creativeById = creatives.stream().collect(toMap(Creative::getId, identity()));
        container.setCreativeByIdMap(creativeById);

        IdentityHashMap<Banner, Creative> creativeByBanner = new IdentityHashMap<>();
        bannerWithCreatives.forEach(banner -> {
            Long creativeId = banner.getCreativeId();
            creativeByBanner.put(banner, creativeById.get(creativeId));
        });
        container.setBannerToCreativeMap(creativeByBanner);
    }

    public <T extends Banner> void fillSitelinkSets(AbstractBannersOperationContainer container,
                                                    List<T> banners) {
        Set<Long> sitelinksSetIds = StreamEx.of(banners)
                .nonNull()
                .select(BannerWithSitelinks.class)
                .map(BannerWithSitelinks::getSitelinksSetId)
                .nonNull()
                .toSet();

        var sitelinkSets = sitelinkSetRepository.get(container.getShard(), container.getClientId(), sitelinksSetIds);
        container.setSitelinkSets(sitelinkSets);
    }

    /**
     * Достаёт турболендинги для баннеров и их сайтлинков, объединяя их в одну мапу
     * (чтобы далее можно было получить турболендинг и для любого баннера, и для любого сайтлинка внутри баннера)
     */
    public <T extends Banner> void fillTurboLandings(AbstractBannersOperationContainer container,
                                                     List<T> banners,
                                                     Map<Long, SitelinkSet> sitelinkSets) {
        Set<Long> bannerTurboLandingsIds = StreamEx.of(banners)
                .nonNull()
                .select(BannerWithTurboLanding.class)
                .map(BannerWithTurboLanding::getTurboLandingId)
                .nonNull()
                .toSet();
        Set<Long> sitelinkTurboLandingsIds = StreamEx.of(sitelinkSets.values())
                .map(SitelinkSet::getSitelinks)
                .flatMap(Collection::stream)
                .map(Sitelink::getTurboLandingId)
                .nonNull()
                .toSet();
        Set<Long> turboLandingsIds = Sets.union(bannerTurboLandingsIds, sitelinkTurboLandingsIds);

        var turboLandings = turboLandingRepository.getTurboLandings(container.getShard(), turboLandingsIds);
        container.setTurboLandings(turboLandings);
    }

    protected void fillContainerPpcProperties(AbstractBannersOperationContainer container) {
        Map<String, String> properties = ppcPropertiesSupport.getByNames(List.of(CPM_GEO_PIN_AUTO_MODERATION.getName(),
                CPM_GEOPRODUCT_AUTO_MODERATION.getName()));
        container.setCpmGeoPinAutoModeration(deserialize(CPM_GEO_PIN_AUTO_MODERATION, properties)
                .orElse(false));
        container.setCpmGeoProductAutoModeration(deserialize(CPM_GEOPRODUCT_AUTO_MODERATION, properties)
                .orElse(false));
    }

    protected <T extends Banner> void fillExistingBigKingImageHashesWithType(AbstractBannersOperationContainer container,
                                                                             List<T> banners) {
        List<String> imageHashes = StreamEx.of(banners)
                .nonNull()
                .select(BannerWithBigKingImage.class)
                .map(BannerWithBigKingImage::getBigKingImageHash)
                .nonNull()
                .collect(toList());

        Map<String, Pair<ImageType, ImageSize>> typedHashes = bannerImageFormatRepository.getExistingImageSizes(
                container.getShard(),
                container.getClientId(),
                imageHashes);
        container.setExistingBigKingImageHashesWithType(typedHashes);
    }

    protected <T extends Banner> void fillExistingImageTypesWithSizes(AbstractBannersOperationContainer container,
                                                                      List<T> banners) {
        List<String> imageHashes = StreamEx.of(banners)
                .nonNull()
                .select(BannerWithImage.class)
                .map(BannerWithImage::getImageHash)
                .nonNull()
                .collect(toList());
        Map<String, Pair<ImageType, ImageSize>> typesWithSizes =
                bannerImageFormatRepository.getExistingImageSizes(container.getShard(), container.getClientId(),
                        imageHashes);
        container.setExistingImageTypesWithSizes(typesWithSizes);
    }

    protected <T extends Banner> void fillImages(AbstractBannersOperationContainer container,
                                                 List<T> banners) {
        List<String> imageHashes = StreamEx.of(banners)
                .nonNull()
                .select(BannerWithBannerImage.class)
                .map(BannerWithBannerImage::getImageHash)
                .nonNull()
                .collect(toList());

        Map<String, BannerImageFormat> bannerImageFormats =
                bannerImageFormatRepository.getBannerImageFormats(container.getShard(), imageHashes);

        container.setBannerImageFormats(bannerImageFormats);
    }

    private void fillMobileAppTrackers(AbstractBannersOperationContainer container) {
        Map<Long, CampaignWithMobileContent> campaigns = container.getCampaignIdToCampaignWithMobileContentMap();
        if (!campaigns.isEmpty()) {
            Set<Long> mobileAppIds = StreamEx.of(campaigns.values())
                    .map(CampaignWithMobileContent::getMobileAppId)
                    .toSet();
            Map<Long, List<MobileAppTracker>> mobileAppIdToTrackersMap = mobileAppRepository.getMobileTrackers(
                    container.getShard(), container.getClientId(), mobileAppIds);
            container.setMobileAppIdToTrackersMap(mobileAppIdToTrackersMap);
        } else {
            container.setMobileAppIdToTrackersMap(Map.of());
        }
    }
}
