package ru.yandex.direct.core.entity.banner.container;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganizationAndPhone;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeHelper;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.clientphone.ClientPhoneIdsByTypeContainer;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organizations.service.OrganizationService;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Suppliers.memoize;
import static ru.yandex.direct.core.entity.banner.container.BannerOperationContainerUtils.needToCheckPermalinkAccessForUpdate;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.selectAppliedChanges;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterValidSubResults;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class BannersUpdateOperationContainerService extends BannersOperationContainerService {

    private final BannerTypedRepository typedRepository;
    private final AdGroupRepository adGroupRepository;
    private final VcardRepository vcardRepository;

    @Autowired
    public BannersUpdateOperationContainerService(CampaignTypedRepository campaignTypedRepository,
                                                  PpcPropertiesSupport ppcPropertiesSupport,
                                                  BannerWithCreativeHelper bannerWithCreativeHelper,
                                                  CreativeService creativeService,
                                                  BannerTypedRepository typedRepository,
                                                  AdGroupRepository adGroupRepository,
                                                  OrganizationService organizationService,
                                                  VcardRepository vcardRepository,
                                                  SitelinkSetRepository sitelinkSetRepository,
                                                  TurboLandingRepository turboLandingRepository,
                                                  PricePackageRepository pricePackageRepository,
                                                  BannerImageRepository bannerImageRepository,
                                                  BannerImageFormatRepository bannerImageFormatRepository,
                                                  ClientPhoneRepository clientPhoneRepository,
                                                  MobileAppRepository mobileAppRepository,
                                                  NetAcl netAcl) {
        super(campaignTypedRepository, ppcPropertiesSupport, bannerWithCreativeHelper, creativeService,
                sitelinkSetRepository, turboLandingRepository, pricePackageRepository, bannerImageRepository, bannerImageFormatRepository,
                clientPhoneRepository, mobileAppRepository, organizationService, netAcl);
        this.typedRepository = typedRepository;
        this.adGroupRepository = adGroupRepository;
        this.vcardRepository = vcardRepository;
    }

    public <T extends Banner> void fillContainers(BannersUpdateOperationContainerImpl container,
                                                  List<ModelChanges<T>> modelChanges) {
        Map<Integer, Long> indexToBannerId = EntryStream.of(modelChanges)
                .mapValues(ModelChanges::getId)
                .filterValues(CommonUtils::isValidId)
                .toMap();
        int shard = container.getShard();

        Map<Long, Long> bannerIdToAdGroupId =
                typedRepository.getBannerIdToAdGroupId(shard, new HashSet<>(indexToBannerId.values()));
        Set<Long> adGroupIds = new HashSet<>(bannerIdToAdGroupId.values());

        Map<Long, AdGroupForBannerOperation> adGroupIdToAdgroup =
                adGroupRepository.getAdGroupsForBannerOperation(container.getShard(), adGroupIds,
                        container.getClientId());

        Map<Integer, AdGroupForBannerOperation> indexToAdGroupMap = getIndexToAdGroupMap(indexToBannerId,
                bannerIdToAdGroupId, adGroupIdToAdgroup);
        initializeAdGroupsAndCampaignMapsInContainers(container, indexToAdGroupMap);

        Map<Long, AdGroupForBannerOperation> bannerIdToAdgroup = EntryStream.of(bannerIdToAdGroupId)
                .mapValues(adGroupIdToAdgroup::get)
                .nonNullValues()
                .toMap();
        container.setBannerIdToAdGroupMap(bannerIdToAdgroup);

        Supplier<Map<Integer, ContentPromotionAdgroupType>> indexToContentPromotionAdGroupType = memoize(() ->
                getIndexToContentPromotionAdGroupTypeMap(container, indexToBannerId, bannerIdToAdGroupId, adGroupIds));
        container.setIndexToContentPromotionAdgroupTypeMap(indexToContentPromotionAdGroupType);

        fillContainerPpcProperties(container);
    }

    public <T extends Banner> void fillContainerOnChangesApplied(BannersUpdateOperationContainerImpl container,
                                                                 Collection<AppliedChanges<T>> validAppliedChanges) {
        var models = mapList(validAppliedChanges, AppliedChanges::getModel);
        fillCreativeMap(container, models);
        fillChangedClientOrganizations(container, validAppliedChanges);
        fillSitelinkSets(container, models);
        fillTurboLandings(container, models, container.getSitelinkSets());
        fillAllVcards(container, validAppliedChanges);
        fillOrganizationPhones(container, validAppliedChanges);
        fillExistingImageFormats(container, models);
    }

    public <T extends Banner> void fillExistingImageFormats(BannersUpdateOperationContainerImpl container,
                                                            List<T> banners) {
        fillExistingImageTypesWithSizes(container, banners);
        fillExistingBigKingImageHashesWithType(container, banners);
        fillImages(container, banners);
    }

    public <T extends Banner> void fillContainerOnModelChangesBeforeApply(
            BannersUpdateOperationContainerImpl container,
            ValidationResult<List<ModelChanges<T>>, Defect> preValidateResult,
            Map<Long, T> models) {
        ValidationResult<List<ModelChanges<T>>, Defect> vrWithValidSubResults =
                filterValidSubResults(preValidateResult);

        if (!vrWithValidSubResults.getValue().isEmpty()) {
            fillExistingImageTypesWithSizesBeforeApply(container, models.values(), vrWithValidSubResults.getValue());
        }
    }

    public <T extends Banner> void fillOrganizationPhones(BannersUpdateOperationContainerImpl container,
                                                          Collection<AppliedChanges<T>> validAppliedChanges) {
        Set<Long> permalinks = extractPermalinks(validAppliedChanges);
        var phoneIdsContainer = fillAllowedPhoneIds(container, permalinks);

        phoneIdsContainer.setAllowOverrideNotManualPhoneWithNoRights(container, netAcl);
        boolean allowOverrideNotManual = phoneIdsContainer.isAllowedOverrideNotManual();

        var permalinksToAccessCheck = extractPermalinksToAccessCheckForUpdate(validAppliedChanges,
                phoneIdsContainer, container, allowOverrideNotManual);
        fillContainerAccessibleOrganizationsPermalinkId(container, permalinksToAccessCheck);
    }


    private <T extends Banner> Set<Long> extractPermalinks(Collection<AppliedChanges<T>> changes) {
        return StreamEx.of(changes)
                .filter(ac -> ac.getModel() instanceof BannerWithOrganizationAndPhone)
                .map(ac -> ac.castModelUp(BannerWithOrganizationAndPhone.class))
                .map(ac -> ac.getNewValue(BannerWithOrganizationAndPhone.PERMALINK_ID))
                .nonNull()
                .toSet();
    }

    /**
     * Возвращает множество пермалинков, у которых нужно будет проверить доступ при обновлении баннера.
     * Доступ нужно проверить, если:
     * <ul>
     * <li>при выключенных фичах {@link ru.yandex.direct.feature.FeatureName#TELEPHONY_ALLOWED}
     * и {@link ru.yandex.direct.feature.FeatureName#UNIVERSAL_CAMPAIGNS_ENABLED_FOR_UAC}
     * если номер изменился на любой
     * </li>
     * <li>
     * при хотя бы одной включенной фиче {@link ru.yandex.direct.feature.FeatureName#TELEPHONY_ALLOWED}
     * или {@link ru.yandex.direct.feature.FeatureName#UNIVERSAL_CAMPAIGNS_ENABLED_FOR_UAC}
     * если номер изменился на ручной или если номер не изменился, организация изменилась
     * и при этом номер в новой организации ручной
     * </li>
     * </ul>
     */
    public static <T extends Banner> Set<Long> extractPermalinksToAccessCheckForUpdate(
            Collection<AppliedChanges<T>> changes,
            ClientPhoneIdsByTypeContainer phoneIdsContainer,
            BannersOperationContainer bannersContainer,
            boolean allowOverrideNotManual
    ) {
        return StreamEx.of(changes)
                .filter(ac -> ac.getModel() instanceof BannerWithOrganizationAndPhone)
                .map(ac -> ac.castModelUp(BannerWithOrganizationAndPhone.class))
                .filter(ac -> needToCheckPermalinkAccessForUpdate(ac,
                        phoneIdsContainer, bannersContainer, allowOverrideNotManual))
                .map(ac -> ac.getNewValue(BannerWithOrganizationAndPhone.PERMALINK_ID))
                .nonNull()
                .toSet();
    }

    private <T extends Banner> void fillAllVcards(AbstractBannersOperationContainer container,
                                                  Collection<AppliedChanges<T>> validAppliedChanges) {
        var appliedChanges = StreamEx.of(validAppliedChanges)
                .filter(t -> t.getModel() instanceof BannerWithVcard)
                .map(t -> t.castModelUp(BannerWithVcard.class))
                .toList();
        StreamEx<Long> oldVcardIds = StreamEx.of(appliedChanges)
                .map(changes -> changes.getOldValue(BannerWithVcard.VCARD_ID))
                .nonNull();
        StreamEx<Long> newVcardIds = StreamEx.of(appliedChanges)
                .map(changes -> changes.getNewValue(BannerWithVcard.VCARD_ID))
                .nonNull();
        Set<Long> vcardIds = StreamEx.of(oldVcardIds)
                .append(newVcardIds)
                .toSet();

        var vcards = vcardRepository.getVcards(container.getShard(), container.getChiefUid(), vcardIds);
        container.setVcardIdToData(listToMap(vcards, Vcard::getId));
    }

    public <T extends Banner> void fillChangedClientOrganizations(
            AbstractBannersOperationContainer container,
            Collection<AppliedChanges<T>> validAppliedChanges) {
        Map<Long, Organization> changedClientOrganizations = getChangedClientOrganizations(container,
                validAppliedChanges);
        container.setClientOrganizations(changedClientOrganizations);
    }

    /**
     * Возвращает организации, которые изменились и на которые клиент имеет права.
     */
    private <T extends Banner> Map<Long, Organization> getChangedClientOrganizations(
            AbstractBannersOperationContainer container,
            Collection<AppliedChanges<T>> bannersChanges) {
        List<Long> changedPermalinkIds = StreamEx.of(
                        selectAppliedChanges(bannersChanges, BannerWithOrganization.class))
                .filter(ac -> ac.changedAndNotDeleted(BannerWithOrganization.PERMALINK_ID))
                .map(ac -> ac.getNewValue(BannerWithOrganization.PERMALINK_ID))
                .nonNull()
                .toList();
        return organizationService.getClientOrganizations(changedPermalinkIds, container.getClientId());
    }

    private Map<Integer, AdGroupForBannerOperation> getIndexToAdGroupMap(
            Map<Integer, Long> indexToBannerId,
            Map<Long, Long> bannerIdToAdGroupId,
            Map<Long, AdGroupForBannerOperation> adGroupsInfo) {

        return EntryStream.of(indexToBannerId)
                .mapValues(bannerIdToAdGroupId::get)
                .nonNullValues()
                .mapValues(adGroupsInfo::get)
                .nonNullValues()
                .toMap();
    }

    /**
     * Извлекаем информацию о типах контента в контент-промоушн-группах из базы
     *
     * @return позиция баннера -> ContentPromotionAdgroupType
     */
    private Map<Integer, ContentPromotionAdgroupType> getIndexToContentPromotionAdGroupTypeMap(
            AbstractBannersOperationContainer container,
            Map<Integer, Long> indexToBannerId,
            Map<Long, Long> bannerIdToAdGroupId,
            Set<Long> adGroupIds) {
        Map<Long, ContentPromotionAdgroupType> adGroupIdToContentPromotionType =
                adGroupRepository.getContentPromotionAdGroupTypesByIds(
                        container.getShard(), container.getClientId(), adGroupIds);
        return EntryStream.of(indexToBannerId)
                .mapValues(bannerIdToAdGroupId::get)
                .nonNullValues()
                .mapValues(adGroupIdToContentPromotionType::get)
                .nonNullValues()
                .toMap();
    }

    private <T extends Banner> void fillExistingImageTypesWithSizesBeforeApply(
            AbstractBannersOperationContainer container,
            Collection<T> banners,
            List<ModelChanges<T>> modelChanges) {

        Set<String> imageHashes = new HashSet<>();
        StreamEx.of(banners)
                .select(BannerWithImage.class)
                .map(BannerWithImage::getImageHash)
                .filter(Objects::nonNull)
                .forEach(imageHashes::add);
        StreamEx.of(modelChanges)
                .map(mc -> mc.castModel(BannerWithImage.class))
                .map(mc -> mc.getPropIfChanged(BannerWithImage.IMAGE_HASH))
                .filter(Objects::nonNull)
                .forEach(imageHashes::add);
        Map<String, Pair<ImageType, ImageSize>> typesWithSizes = bannerImageFormatRepository.getExistingImageSizes(
                container.getShard(),
                container.getClientId(),
                imageHashes
        );
        container.setExistingTypesWithSizesBeforeApply(typesWithSizes);
    }
}
