package ru.yandex.direct.core.entity.banner.repository;

import java.sql.PreparedStatement;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.EnumType;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.Query;
import org.jooq.Record2;
import org.jooq.Table;
import org.jooq.TransactionalRunnable;
import org.jooq.UpdateSetMoreStep;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.model.BannerImageOpts;
import ru.yandex.direct.core.entity.banner.model.BannerWithLanguage;
import ru.yandex.direct.core.entity.banner.type.system.BannerWithSystemFieldsMappings;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.dbschema.ppc.enums.BannerDisplayHrefsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BannerTurbolandingsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersPhoneflag;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatussitelinksmoderate;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsStatusmetricacontrol;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.ImagesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusshowsforecast;
import ru.yandex.direct.dbschema.ppc.enums.RedirectCheckQueueObjectType;
import ru.yandex.direct.dbschema.ppc.tables.BannerDisplayHrefs;
import ru.yandex.direct.dbschema.ppc.tables.BannerTurboGalleries;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.DeletedBannersRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.isnull;
import static org.jooq.impl.DSL.when;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.core.entity.banner.repository.mapper.TextBannerMappings.updateGeoFlagInOpts;
import static ru.yandex.direct.core.entity.banner.type.system.BannerWithSystemFieldsMappings.statusBsSyncedToDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AGGREGATOR_DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_CONTENT_PROMOTION;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_CONTENT_PROMOTION_VIDEO;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MINUS_GEO;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PHONES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PRICES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PUBLISHER;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDING_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_USER_FLAGS_UPDATES;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_DEAD_DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.DELETED_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.MEDIAPLAN_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.REDIRECT_CHECK_QUEUE;
import static ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate.New;
import static ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate.Ready;
import static ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate.Sent;
import static ru.yandex.direct.dbschema.ppc.tables.BannerDisplayHrefTexts.BANNER_DISPLAY_HREF_TEXTS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImages.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerLeadformAttributes.BANNER_LEADFORM_ATTRIBUTES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerMulticardSets.BANNER_MULTICARD_SETS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerMulticards.BANNER_MULTICARDS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerTurboApps.BANNER_TURBO_APPS;
import static ru.yandex.direct.dbschema.ppc.tables.Domains.DOMAINS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class BannerCommonRepository {

    private static final Logger LOGGER = LoggerFactory.getLogger(BannerCommonRepository.class);
    private static final int MAX_ATTEMPTS_TO_RETRY = 3;

    private final DslContextProvider ppcDslContextProvider;

    public BannerCommonRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;
    }

    public Map<Long, Boolean> getEffectiveStopByMonitoringStatusByBannerIds(int shard, Collection<Long> bids) {
        String status = "IF (`camp_options`.`statusMetricaControl` = 'Yes' " +
                "AND `bs_dead_domains`.`domain_id` IS NOT NULL, TRUE, FALSE)";

        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID, DSL.field(status, Boolean.class))
                .from(BANNERS)
                .leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(BANNERS.CID))
                .leftJoin(DOMAINS).on(DOMAINS.DOMAIN.eq(BANNERS.DOMAIN)) // DOMAIN_ID?
                .leftJoin(BS_DEAD_DOMAINS).on(BS_DEAD_DOMAINS.DOMAIN_ID.eq(DOMAINS.DOMAIN_ID))
                .where(BANNERS.BID.in(bids))
                .fetchMap(BANNERS.BID, Record2::component2);
    }

    /**
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/BS/CheckUrlAvailability.pm?rev=r7629508#L74">Reference</a>
     *
     * @return список баннеров, остановленных с помощью мониторинга БК (bs_dead_domains)
     */
    public List<Long> getBannersStoppedByMonitoringByCampaignId(int shard, Long campaignId) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(BANNERS.CID))
                .join(DOMAINS).on(DOMAINS.DOMAIN.eq(BANNERS.DOMAIN))
                .join(BS_DEAD_DOMAINS).on(BS_DEAD_DOMAINS.DOMAIN_ID.eq(DOMAINS.DOMAIN_ID))
                .where(CAMPAIGNS.CID.eq(campaignId))
                .and(CAMPAIGNS.TYPE.in(mapList(CampaignTypeKinds.ALLOW_DOMAIN_MONITORING, CampaignType::toSource)))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes))
                .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes))
                .and(CAMP_OPTIONS.STATUS_METRICA_CONTROL.eq(CampOptionsStatusmetricacontrol.Yes))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .fetch(BANNERS.BID);
    }

    /**
     * Блокирует баннеры для последующего обновления.
     *
     * @param dslContext контекст транзакции
     * @param bids       id баннеров
     * @return id заблокированных баннеров
     */
    public List<Long> lockBannersForUpdate(DSLContext dslContext, Collection<Long> bids) {
        return dslContext
                .select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.BID.in(bids))
                .forUpdate()
                .fetch(BANNERS.BID);
    }

    public void deleteBanners(int shard, Collection<Long> bannerIds) {
        deleteBanners(ppcDslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Удаление объявлений по Id из BD.
     *
     * @param bannerIds список id баннеров
     * @param context   контекст
     */
    public void deleteBanners(DSLContext context, Collection<Long> bannerIds) {
        List<Query> toDelete = new ArrayList<>();

        // удалить сам баннер
        toDelete.add(context.deleteFrom(BANNERS_CONTENT_PROMOTION_VIDEO)
                .where(BANNERS_CONTENT_PROMOTION_VIDEO.BID.in(bannerIds)));
        toDelete.add(context.deleteFrom(BANNERS_CONTENT_PROMOTION).where(BANNERS_CONTENT_PROMOTION.BID.in(bannerIds)));
        toDelete.add(context.deleteFrom(BANNERS_MOBILE_CONTENT).where(BANNERS_MOBILE_CONTENT.BID.in(bannerIds)));
        toDelete.add(context.deleteFrom(BANNERS_PERFORMANCE).where(BANNERS_PERFORMANCE.BID.in(bannerIds)));
        toDelete.add(context.deleteFrom(BANNERS).where(BANNERS.BID.in(bannerIds)));
        toDelete.add(context.deleteFrom(BANNERS_MINUS_GEO).where(BANNERS_MINUS_GEO.BID.in(bannerIds)));
        toDelete.add(context.deleteFrom(BANNERS_INTERNAL).where(BANNERS_INTERNAL.BID.in(bannerIds)));
        //удалить картинки
        toDelete.add(context
                .deleteFrom(BANNER_IMAGES)
                .where(BANNER_IMAGES.BID.in(bannerIds)));
        // удалить графические объявления
        toDelete.add(context.deleteFrom(IMAGES)
                .where(IMAGES.BID.in(bannerIds)));
        // удалить дополнения
        toDelete.add(context.deleteFrom(BANNERS_ADDITIONS)
                .where(BANNERS_ADDITIONS.BID.in(bannerIds)));
        // удалить отображаемые урлы
        toDelete.add(context
                .deleteFrom(BANNER_DISPLAY_HREFS)
                .where(BANNER_DISPLAY_HREFS.BID.in(bannerIds)));
        // удалить кастомные тексты отображаемых урлов
        toDelete.add(context
                .deleteFrom(BANNER_DISPLAY_HREF_TEXTS)
                .where(BANNER_DISPLAY_HREF_TEXTS.BID.in(bannerIds)));
        // удалить привязку турболендингов к баннерам
        toDelete.add(context.deleteFrom(BANNER_TURBOLANDINGS).where(BANNER_TURBOLANDINGS.BID.in(bannerIds)));
        // удалить дополнительные параметры ссылок турболендингов баннеров
        toDelete.add(context
                .deleteFrom(BANNER_TURBOLANDING_PARAMS)
                .where(BANNER_TURBOLANDING_PARAMS.BID.in(bannerIds)));
        // удалить цены на баннерах
        toDelete.add(context
                .deleteFrom(BANNER_PRICES)
                .where(BANNER_PRICES.BID.in(bannerIds)));
        // удалить привязки баннеров к организациям
        toDelete.add(context
                .deleteFrom(BANNER_PERMALINKS)
                .where(BANNER_PERMALINKS.BID.in(bannerIds)));
        // удалить привязки телефонов к баннерам
        toDelete.add(context
                .deleteFrom(BANNER_PHONES)
                .where(BANNER_PHONES.BID.in(bannerIds)));
        // удалить ссылки на турбо-галерею
        toDelete.add(context
                .deleteFrom(BannerTurboGalleries.BANNER_TURBO_GALLERIES)
                .where(BannerTurboGalleries.BANNER_TURBO_GALLERIES.BID.in(bannerIds)));
        // удалить домен-аггрегатор
        toDelete.add(context
                .deleteFrom(AGGREGATOR_DOMAINS)
                .where(AGGREGATOR_DOMAINS.BID.in(bannerIds)));
        // удалить турбо-аппы
        toDelete.add(context
                .deleteFrom(BANNER_TURBO_APPS)
                .where(BANNER_TURBO_APPS.BID.in(bannerIds)));
        // удалить атрибуты лидформ
        toDelete.add(context
                .deleteFrom(BANNER_LEADFORM_ATTRIBUTES)
                .where(BANNER_LEADFORM_ATTRIBUTES.BID.in(bannerIds)));
        // удалить информацию об обновлении пользователем флагов
        toDelete.add(context
                .deleteFrom(BANNER_USER_FLAGS_UPDATES)
                .where(BANNER_USER_FLAGS_UPDATES.BID.in(bannerIds)));
        // удалить мультибаннер
        toDelete.add(context.deleteFrom(BANNER_MULTICARD_SETS).where(BANNER_MULTICARD_SETS.BID.in(bannerIds)));
        toDelete.add(context.deleteFrom(BANNER_MULTICARDS).where(BANNER_MULTICARDS.BID.in(bannerIds)));

        context.batch(toDelete).execute();

        InsertValuesStep2<DeletedBannersRecord, Long, LocalDateTime> batch = context
                .insertInto(DELETED_BANNERS, DELETED_BANNERS.BID, DELETED_BANNERS.DELETE_TIME);
        bannerIds.forEach(id -> batch.values(id, LocalDateTime.now()));
        batch.onDuplicateKeyIgnore().execute();

        context.update(MEDIAPLAN_BANNERS)
                .set(MEDIAPLAN_BANNERS.SOURCE_BID, 0L)
                .where(MEDIAPLAN_BANNERS.SOURCE_BID.in(bannerIds));
    }

    public void resetStatusBsSyncedByIds(int shard, Collection<Long> bannerIds) {
        resetStatusBsSyncedByIds(ppcDslContextProvider.ppc(shard), bannerIds);
    }

    public void resetStatusBsSyncedByIds(DSLContext dslContext, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        dslContext.update(BANNERS)
                .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                .set(BANNERS.LAST_CHANGE, BANNERS.LAST_CHANGE)
                .where(BANNERS.BID.in(bannerIds))
                .execute();
    }

    //используется в модерации
    public void resetStatusBsSynced(DSLContext dslContext, Collection<Long> bids) {
        /*
             # phrases.statusBsSynced здесь обновляется только чтобы отправить потенциально новый баннер в БК вместе
             с целым условием
             # подробности см. в DIRECT-33440, после DIRECT-30677 можно будет выпилить
             # comment expires: 2015-12-31
         */

        dslContext
                .update(BANNERS.join(PHRASES).using(BANNERS.PID))
                .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                .set(PHRASES.STATUS_BS_SYNCED, PhrasesStatusbssynced.No)
                .set(BANNERS.LAST_CHANGE, BANNERS.LAST_CHANGE)
                .where(BANNERS.BID.in(bids))
                .execute();
    }

    /**
     * Обновить statusBsSynced и lastChange у баннеров, привязанных к указанным vcard-ам
     *
     * @return - количество изменённых строк
     */
    public int updateStatusBsSyncedAndLastChangeByVcardId(int shard, Collection<Long> vcardIds, StatusBsSynced status) {
        return ppcDslContextProvider.ppc(shard)
                .update(BANNERS)
                .set(BANNERS.STATUS_BS_SYNCED, statusBsSyncedToDb(status))
                .set(BANNERS.LAST_CHANGE, DSL.currentLocalDateTime())
                .where(BANNERS.VCARD_ID.in(vcardIds))
                .execute();
    }

    /**
     * Массовое обновление языка баннеров.
     */
    public void updateBannersLanguages(DSLContext dslContext,
                                       Collection<AppliedChanges<BannerWithLanguage>> bannersChanges) {
        JooqUpdateBuilder<BannersRecord, BannerWithLanguage> updateBuilder =
                new JooqUpdateBuilder<>(BANNERS.BID, bannersChanges);

        updateBuilder.processProperty(BannerWithLanguage.LANGUAGE, BANNERS.LANGUAGE,
                BannerWithSystemFieldsMappings::languageToDb);

        dslContext
                .update(BANNERS)
                .set(updateBuilder.getValues())
                .set(BANNERS.LAST_CHANGE, BANNERS.LAST_CHANGE)
                .where(BANNERS.BID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    /**
     * Возвращает список, состоящий из не более limit максимальных id баннеров, строго меньших чем upperBoundBid.
     *
     * @param shard         шард
     * @param upperBoundBid верхняя граница id баннеров
     * @param limit         ограничение на длину возвращаемого списка
     */
    public List<Long> getBannerIdsLessThan(int shard, Long upperBoundBid, int limit) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.BID.lessThan(upperBoundBid))
                .orderBy(BANNERS.BID.desc())
                .limit(limit)
                .fetch(BANNERS.BID);
    }

    /**
     * Обновить statusBsSynced для баннеров с указанным id кампании из списка campaignIds
     *
     * @param shard       шард
     * @param campaignIds список id кампаний
     * @param status      новый статус
     * @return количество изменённых строк
     */
    public int updateStatusBsSyncedByCampaignIds(int shard, Collection<Long> campaignIds, StatusBsSynced status) {
        return updateStatusBsSyncedByCampaignIds(ppcDslContextProvider.ppc(shard).configuration(),
                campaignIds, false, status);
    }

    /**
     * Обновить statusBsSynced для баннеров с указанным id кампании из списка campaignIds
     *
     * @param conf        кофиг
     * @param campaignIds список id кампаний
     * @param status      новый статус
     * @return количество изменённых строк
     */
    public int updateStatusBsSyncedByCampaignIds(Configuration conf, Collection<Long> campaignIds,
                                                 boolean updateLastChange, StatusBsSynced status) {
        if (campaignIds.isEmpty()) {
            return 0;
        }

        UpdateSetMoreStep<BannersRecord> updateQuery = DSL.using(conf)
                .update(BANNERS)
                .set(BANNERS.STATUS_BS_SYNCED, statusBsSyncedToDb(status));
        if (!updateLastChange) {
            updateQuery.set(BANNERS.LAST_CHANGE, BANNERS.LAST_CHANGE);
        }

        return updateQuery
                .where(BANNERS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Обновить statusBsSynced у баннеров, принадлежащих указанным adgroup-ам
     *
     * @return - количество изменённых строк
     */
    public int updateStatusBsSyncedByAdgroupId(int shard, Collection<Long> adGroupIds, StatusBsSynced status) {
        return updateStatusBsSyncedByAdgroupId(ppcDslContextProvider.ppc(shard), adGroupIds, status);
    }

    /**
     * Обновить statusBsSynced у баннеров, принадлежащих указанным adgroup-ам
     *
     * @return - количество изменённых строк
     */
    public int updateStatusBsSyncedByAdgroupId(DSLContext context, Collection<Long> adGroupIds, StatusBsSynced status) {
        if (adGroupIds.isEmpty()) {
            return 0;
        }
        return context
                .update(BANNERS)
                .set(BANNERS.STATUS_BS_SYNCED, statusBsSyncedToDb(status))
                .set(BANNERS.LAST_CHANGE, BANNERS.LAST_CHANGE)
                .where(BANNERS.PID.in(adGroupIds))
                .execute();
    }

    public int updateBannersLastChangeByCampaignIds(DSLContext context, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return 0;
        }

        return context
                .update(BANNERS)
                .set(BANNERS.LAST_CHANGE, DSL.currentLocalDateTime())
                .where(BANNERS.CID.in(campaignIds))
                .execute();
    }

    public void resetStatusActiveAndArchiveStatusShowsForecast(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }
        executeTransactionWithRetries(shard, conf -> {
            conf.dsl().update(BANNERS)
                    .set(BANNERS.LAST_CHANGE, DSL.currentLocalDateTime())
                    .set(BANNERS.STATUS_ACTIVE, BannersStatusactive.No)
                    .where(BANNERS.CID.in(campaignIds))
                    .orderBy(BANNERS.BID)
                    .execute();
            conf.dsl().update(PHRASES)
                    .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE) // иначе сбросится в CURRENT_TIMESTAMP
                    .set(PHRASES.STATUS_SHOWS_FORECAST, PhrasesStatusshowsforecast.Archived)
                    .where(PHRASES.CID.in(campaignIds))
                    .orderBy(PHRASES.PID)
                    .execute();
        });
    }

    // Перенесено из https://a.yandex-team.ru/arcadia/direct/perl/protected/Common.pm?rev=r9287185#L3165
    public void unarchiveStatusBsSyncedAndStatusShowsForecast(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }
        executeTransactionWithRetries(shard, conf -> {
            conf.dsl().update(BANNERS)
                    .set(BANNERS.LAST_CHANGE, DSL.currentLocalDateTime())
                    .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                    .where(BANNERS.CID.in(campaignIds))
                    .orderBy(BANNERS.BID)
                    .execute();
            conf.dsl().update(PHRASES)
                    .set(PHRASES.LAST_CHANGE, PHRASES.LAST_CHANGE) // иначе сбросится в CURRENT_TIMESTAMP
                    .set(PHRASES.STATUS_BS_SYNCED, PhrasesStatusbssynced.No)
                    .set(PHRASES.STATUS_SHOWS_FORECAST, PhrasesStatusshowsforecast.New)
                    .where(PHRASES.CID.in(campaignIds))
                    .orderBy(PHRASES.PID)
                    .execute();
        });
    }

    private void executeTransactionWithRetries(int shard, TransactionalRunnable runnable) {
        for (int tries = 0; ; tries++) {
            try {
                ppcDslContextProvider.ppc(shard).transaction(runnable);
                break;
            } catch (DataAccessException e) {
                LOGGER.error("error in executeTransactionWithRetries (tries = {}): {}", tries, e);
                if (tries >= MAX_ATTEMPTS_TO_RETRY || !e.getMessage().contains("Deadlock found")) {
                    throw e;
                }
            }
        }
    }

    public Map<Long, Long> getBidsForBannerIds(List<Long> bannerIds, int shard) {
        Map<Long, Long> bannerIdToBid = StreamEx.of(ppcDslContextProvider.ppc(shard)
                        .select(BANNERS.BANNER_ID, BANNERS.BID)
                        .from(BANNERS)
                        .where(BANNERS.BANNER_ID.in(bannerIds))
                        .fetch())
                .toMap(r -> r.getValue(BANNERS.BANNER_ID), r -> r.getValue(BANNERS.BID));

        // если для каких-то bannerID не нашли bid - пытаемся поискать среди картиночных баннеров
        List<Long> missing =
                bannerIds.stream().filter(bannerId -> !bannerIdToBid.containsKey(bannerId)).collect(toList());
        if (!missing.isEmpty()) {
            bannerIdToBid.putAll(
                    StreamEx.of(ppcDslContextProvider.ppc(shard)
                                    .select(BANNER_IMAGES.BANNER_ID, BANNER_IMAGES.IMAGE_ID)
                                    .from(BANNER_IMAGES)
                                    .where(BANNER_IMAGES.BANNER_ID.in(missing))
                                    .fetch())
                            .toMap(r -> r.getValue(BANNER_IMAGES.BANNER_ID), r -> r.getValue(BANNER_IMAGES.IMAGE_ID))
            );
        }

        return bannerIdToBid;
    }

    /**
     * Добавить баннеры в очередь на проверку редиректа:
     * для добавления необходимо заполненное поле Banner.id
     */
    public void addToRedirectCheckQueue(int shard, List<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        final DSLContext context = ppcDslContextProvider.ppc(shard);
        context.connection(connection -> {
            String sql = String.format("REPLACE INTO %s(%s, %s) VALUES %s",
                    context.render(REDIRECT_CHECK_QUEUE),
                    context.render(REDIRECT_CHECK_QUEUE.OBJECT_ID),
                    context.render(REDIRECT_CHECK_QUEUE.OBJECT_TYPE),
                    StreamEx.of(bannerIds).map(b -> "(?, ?)").joining(", "));

            PreparedStatement statement = connection.prepareStatement(sql);

            for (int i = 0; i < bannerIds.size(); i++) {
                int parameterIndex = 2 * i + 1; // в sql запрос передаётся 2 значения, счёт параметров начинается с 1
                statement.setLong(parameterIndex, bannerIds.get(i));
                statement.setString(parameterIndex + 1, RedirectCheckQueueObjectType.banner.getLiteral());
            }

            statement.execute();
        });
    }

    /**
     * Сбросить флаг status_bs_synced и обновить geoflag для баннеров (См. protected/Direct/Model/Manager
     * .pm::_do_update_banners)
     * <p>
     * Обновление geoflag-а происходит только если он был изменен
     */
    public void resetStatusBsSyncedAndUpdateGeoFlagByIds(int shard, Map<Long, Boolean> bannerIdToNewGeoFlag) {
        if (bannerIdToNewGeoFlag.isEmpty()) {
            return;
        }

        ppcDslContextProvider.ppc(shard).transaction(config -> {
            Map<Long, String> bannerIdToOldOpts = config.dsl().select(BANNERS.BID, BANNERS.OPTS)
                    .from(BANNERS)
                    .where(BANNERS.BID.in(bannerIdToNewGeoFlag.keySet()))
                    .forUpdate()
                    .fetchMap(BANNERS.BID, BANNERS.OPTS);

            Preconditions.checkState(!bannerIdToOldOpts.isEmpty());

            Map<Long, String> bannerIdToNewOpts = bannerIdToOldOpts.entrySet()
                    .stream()
                    .collect(
                            toMap(
                                    Map.Entry::getKey,
                                    // Если геофлаг не изменился (т.е. newGeoFlag == null), то оставляем старый opts
                                    e -> Optional.ofNullable(bannerIdToNewGeoFlag.get(e.getKey()))
                                            .map(newGeoFlag -> updateGeoFlagInOpts(e.getValue(), newGeoFlag))
                                            .orElse(e.getValue())));

            config.dsl().update(BANNERS)
                    .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                    .set(BANNERS.OPTS, JooqMapperUtils.makeCaseStatement(BANNERS.BID, BANNERS.OPTS, bannerIdToNewOpts))
                    .set(BANNERS.LAST_CHANGE, BANNERS.LAST_CHANGE)
                    .where(BANNERS.BID.in(bannerIdToNewOpts.keySet()))
                    .execute();
        });
    }

    /**
     * Возвращает по bid BannerID картинки, если
     * - в таблице banner_images существует запись с указанным bid
     * - поле banner_images.statusModerate равно Yes
     * - поле banner_images.statusShow равно Yes
     * - поле banner_images.BannerID не равно 0
     * - нет флага single_ad_to_bs в banner_images.opts
     *
     * @param shard шард
     * @param bid   id баннера
     * @return BannerID или null, если баннера с указанным bid не существует
     */
    public Long getBsBannerIdForAiming(int shard, long bid) {
        DSLContext ctxt = ppcDslContextProvider.ppc(shard);

        Field<Long> bannerIdField = DSL.field("BannerID", Long.class);
        return ctxt.select(isnull(BANNER_IMAGES.BANNER_ID, BANNERS.BANNER_ID).as(bannerIdField))
                .from(BANNERS)
                .leftJoin(BANNER_IMAGES).on(BANNERS.BID.eq(BANNER_IMAGES.BID)
                        .and(BANNER_IMAGES.STATUS_MODERATE.eq(BannerImagesStatusmoderate.Yes))
                        .and(BANNER_IMAGES.STATUS_SHOW.eq(BannerImagesStatusshow.Yes))
                        .and(BANNER_IMAGES.BANNER_ID.notEqual(0L))
                        .and(BANNER_IMAGES.OPTS.isNull()
                                .or(BANNER_IMAGES.OPTS.notContains(BannerImageOpts.SINGLE_AD_TO_BS.getTypedValue()))))
                .where(BANNERS.BID.eq(bid))
                .fetchOne(bannerIdField);
    }

    /**
     * Сброс статусов модерации и синхронизации с БК у баннеров и их дополнений
     * при изменении текста фраз. Должен применяться к шаблонным баннерам.
     * Статус модерации в таблице banners_performance сбрасывается, только если
     * баннер текстовый, то есть по факту если это видео-дополнение.
     */
    public void dropTemplateBannersStatusesOnKeywordsChange(Configuration conf, Collection<Long> bannerIds) {
        try (DSLContext ctx = DSL.using(conf)) {
            ctx
                    .update(BANNERS
                            .leftJoin(BANNER_IMAGES).on(BANNER_IMAGES.BID.eq(BANNERS.BID))
                            .leftJoin(BannerDisplayHrefs.BANNER_DISPLAY_HREFS).on(BannerDisplayHrefs.BANNER_DISPLAY_HREFS.BID.eq(BANNERS.BID))
                            .leftJoin(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.BID.eq(
                                    DSL.choose(BANNERS.BANNER_TYPE)
                                            .when(BannersBannerType.text, BANNERS.BID)
                                            .otherwise((Long) null))))
                    .set(BANNERS.STATUS_MODERATE, Ready)
                    .set(BANNERS.STATUS_POST_MODERATE, DSL.choose(BANNERS.STATUS_POST_MODERATE)
                            .when(BannersStatuspostmoderate.Rejected, BannersStatuspostmoderate.Rejected)
                            .otherwise(BannersStatuspostmoderate.No))
                    .set(BANNERS.PHONEFLAG,
                            when(BANNERS.VCARD_ID.isNotNull(), BannersPhoneflag.Ready)
                                    .otherwise(BannersPhoneflag.New))
                    .set(BANNERS.STATUS_SITELINKS_MODERATE,
                            when(BANNERS.SITELINKS_SET_ID.isNotNull(), BannersStatussitelinksmoderate.Ready)
                                    .otherwise(BannersStatussitelinksmoderate.New))
                    .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                    .set(BannerDisplayHrefs.BANNER_DISPLAY_HREFS.STATUS_MODERATE,
                            BannerDisplayHrefsStatusmoderate.Ready)
                    .set(BANNER_IMAGES.STATUS_MODERATE, BannerImagesStatusmoderate.Ready)
                    .set(BANNERS_PERFORMANCE.STATUS_MODERATE, BannersPerformanceStatusmoderate.Ready)
                    .where(BANNERS.BID.in(bannerIds))
                    .and(BANNERS.STATUS_MODERATE.notEqual(New))
                    .and(BANNERS.STATUS_ARCH.notEqual(BannersStatusarch.Yes))
                    .execute();
        }
    }

    /**
     * Сброс статусов модерации у баннеров и их дополнений
     * Переводим Ready в Sent чтобы потом перевести обратно в Ready, нужно чтобы ess увидел событие
     *
     * @see <a href=https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Common.pm?rev=r9287185#L4409>
     * Common::_fix_statusModerate_for_unarchived_</a>
     */
    public void fixStatusModerateForUnarchivedObjects(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        ppcDslContextProvider.ppc(shard)
                .update(BANNERS)
                .set(BANNERS.LAST_CHANGE, BANNERS.LAST_CHANGE)
                .set(BANNERS.STATUS_MODERATE,
                        when(BANNERS.STATUS_MODERATE.eq(Ready), Sent)
                                .otherwise(BANNERS.STATUS_MODERATE))
                .set(BANNERS.PHONEFLAG,
                        when(BANNERS.VCARD_ID.isNotNull()
                                        .and(BANNERS.PHONEFLAG.eq(BannersPhoneflag.Ready)),
                                BannersPhoneflag.Sent)
                                .otherwise(BANNERS.PHONEFLAG))
                .set(BANNERS.STATUS_SITELINKS_MODERATE,
                        when(BANNERS.SITELINKS_SET_ID.isNotNull()
                                        .and(BANNERS.STATUS_SITELINKS_MODERATE
                                                .eq(BannersStatussitelinksmoderate.Ready)),
                                BannersStatussitelinksmoderate.Sent)
                                .otherwise(BANNERS.STATUS_SITELINKS_MODERATE))
                .where(BANNERS.BID.in(bannerIds))
                .orderBy(BANNERS.BID)
                .execute();

        updateStatusModerate(shard, bannerIds, BANNER_IMAGES, BANNER_IMAGES.STATUS_MODERATE, BANNER_IMAGES.BID,
                BannerImagesStatusmoderate.Ready, BannerImagesStatusmoderate.Sent);

        updateStatusModerate(shard, bannerIds, IMAGES, IMAGES.STATUS_MODERATE, IMAGES.BID,
                ImagesStatusmoderate.Ready, ImagesStatusmoderate.Sent);

        updateStatusModerate(shard, bannerIds, BANNERS_PERFORMANCE, BANNERS_PERFORMANCE.STATUS_MODERATE,
                BANNERS_PERFORMANCE.BID,
                BannersPerformanceStatusmoderate.Ready, BannersPerformanceStatusmoderate.Sent);

        updateStatusModerate(shard, bannerIds, BANNER_DISPLAY_HREFS, BANNER_DISPLAY_HREFS.STATUS_MODERATE,
                BANNER_DISPLAY_HREFS.BID,
                BannerDisplayHrefsStatusmoderate.Ready, BannerDisplayHrefsStatusmoderate.Sent);

        updateStatusModerate(shard, bannerIds, BANNER_TURBOLANDINGS, BANNER_TURBOLANDINGS.STATUS_MODERATE,
                BANNER_TURBOLANDINGS.BID,
                BannerTurbolandingsStatusmoderate.Ready, BannerTurbolandingsStatusmoderate.Sent);

        updateStatusModerate(shard, bannerIds, PHRASES, PHRASES.STATUS_MODERATE, PHRASES.BID,
                PhrasesStatusmoderate.Ready, PhrasesStatusmoderate.Sent);
    }

    @SuppressWarnings("unchecked")
    private void updateStatusModerate(int shard, Collection<Long> bannerIds, Table table, Field fieldStatus,
                                      Field<Long> fieldBid, EnumType statusFrom, EnumType statusTo) {
        ppcDslContextProvider.ppc(shard)
                .update(table)
                .set(fieldStatus, statusTo)
                .where(fieldBid.in(bannerIds))
                .and(fieldStatus.eq(statusFrom))
                .orderBy(fieldBid)
                .execute();
    }

    /**
     * Сброс статуса остановленности баннеров внутренней рекламы урл-мониторингом
     */
    public void dropInternalBannersStoppedByUrlMonitoring(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        ppcDslContextProvider.ppc(shard)
                .update(BANNERS_INTERNAL)
                .set(BANNERS_INTERNAL.IS_STOPPED_BY_URL_MONITORING, booleanToLong(false))
                .where(BANNERS_INTERNAL.BID.in(bannerIds))
                .execute();
    }

    /**
     * Найти URL последнего созданного объявления с URL среди переданных кампаний
     *
     * @param shard       — шард пользователя
     * @param campaignIds — коллекция id кампаний
     * @return URL
     */
    public String getLatestBannerUrlForCampaigns(int shard, Collection<Long> campaignIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.HREF)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds)
                        .and(BANNERS.HREF.isNotNull()))
                .orderBy(BANNERS.BID.desc())
                .limit(1)
                .fetchOne(BANNERS.HREF);
    }

    /**
     * @return отображение: идентификатор кампании -> множество ссылок, установленных на баннерах этой кампании
     * </p>
     * (!) Если у кампании из {@code campaignIds} ни на одном баннере нет ссылок, то этой кампании не будет
     * в результирующем отображении
     */
    public Map<Long, Set<String>> getHrefsByCampaignId(int shard, Collection<Long> campaignIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.CID, BANNERS.HREF)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds).and(BANNERS.HREF.isNotNull()))
                .fetchGroups(BANNERS.CID, BANNERS.HREF)
                .entrySet()
                .stream()
                .collect(toMap(Map.Entry::getKey, e -> new HashSet<>(e.getValue())));
    }

    /**
     * Получение ссылки по id баннера
     *
     * @return ссылка баннера или null, если ссылки нет в banners.href или bannerId невалидно
     */
    public String getHref(int shard, @Nullable Long bannerId) {
        if (!isValidId(bannerId)) {
            return null;
        }
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.HREF)
                .from(BANNERS)
                .where(BANNERS.BID.eq(bannerId))
                .fetchOne(BANNERS.HREF);
    }

    /**
     * Получение zen_publisher_ids по списку id баннеров
     */
    public Map<Long, String> getZenPublisherIdsByBids(int shard, Collection<Long> bids) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNER_PUBLISHER.BID, BANNER_PUBLISHER.ZEN_PUBLISHER_ID)
                .from(BANNER_PUBLISHER)
                .where(BANNER_PUBLISHER.BID.in(bids))
                .fetchMap(BANNER_PUBLISHER.BID, BANNER_PUBLISHER.ZEN_PUBLISHER_ID);
    }

    /**
     * Возвращает типы баннеров по их идентификаторам
     *
     * @param shard     шард
     * @param bannerIds идентификаторы баннеров
     * @return map id баннера -> тип баннера
     */
    public Map<Long, BannersBannerType> getBannerTypesByIds(int shard, Collection<Long> bannerIds) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.BANNER_TYPE)
                .from(BANNERS)
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(r -> r.getValue(BANNERS.BID), r -> r.getValue(BANNERS.BANNER_TYPE));
    }

    public Map<Long, List<Long>> getCreativeIdToBidsByCampaignId(int shard, Long cid) {
        return ppcDslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS_PERFORMANCE.CREATIVE_ID)
                .from(BANNERS)
                .leftJoin(BANNERS_PERFORMANCE)
                .on(BANNERS.BID.eq(BANNERS_PERFORMANCE.BID))
                .where(BANNERS.CID.eq(cid))
                .fetchGroups(r -> r.getValue(BANNERS_PERFORMANCE.CREATIVE_ID), r -> r.getValue(BANNERS.BID));
    }
}
