package ru.yandex.direct.core.entity.banner.repository;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerGeoLegalFlags;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.StatusModerateBannerPage;
import ru.yandex.direct.core.entity.domain.model.CidAndDomain;
import ru.yandex.direct.dbschema.ppc.enums.BannersMinusGeoType;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersMinusGeoRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.flatMapping;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.jooq.impl.DSL.concat;
import static org.jooq.impl.DSL.groupConcat;
import static org.jooq.impl.DSL.name;
import static org.jooq.impl.DSL.nullif;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTO_MODERATE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MINUS_GEO;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.MODERATE_BANNER_PAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.POST_MODERATE;
import static ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate.New;
import static ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate.Ready;

@Repository
public class BannerModerationRepository {

    public static final Field<String> COLON_SEPARATOR = DSL.field("':'", String.class);
    public static final Field<String> CONCATENATED_MINUS_GEO_WIth_TYPE =
            concat(BANNERS_MINUS_GEO.TYPE, COLON_SEPARATOR, BANNERS_MINUS_GEO.MINUS_GEO)
                    .as(name("concatenatedMinusGeoWithType"));
    public static final Field<String> FLAGS_WITHOUT_BLANKS = nullif(BANNERS.FLAGS, "").as(name("flags_with_nulls"));

    private final DslContextProvider dslContextProvider;

    @Autowired
    public BannerModerationRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Добавляет записи в таблицу ppc.banners_minus_geo.
     * В случае если запись с таким ключом уже есть, эта запись будет обновлена.
     */
    public void insertMinusGeo(Configuration conf, Map<Long, List<Long>> bannersMinusGeo) {
        if (bannersMinusGeo.isEmpty()) {
            return;
        }

        InsertValuesStep2<BannersMinusGeoRecord, Long, String> insertBuilder = conf.dsl()
                .insertInto(BANNERS_MINUS_GEO)
                .columns(BANNERS_MINUS_GEO.BID, BANNERS_MINUS_GEO.MINUS_GEO);
        bannersMinusGeo.forEach((bannerId, minusGeo) ->
                insertBuilder.values(bannerId, minusGeo.stream().map(Object::toString).collect(joining(","))));
        insertBuilder
                .onDuplicateKeyUpdate()
                .set(BANNERS_MINUS_GEO.MINUS_GEO, DSL.field("values({0})", BANNERS_MINUS_GEO.MINUS_GEO.getDataType(),
                        BANNERS_MINUS_GEO.MINUS_GEO))
                .set(BANNERS_MINUS_GEO.TYPE, BannersMinusGeoType.current)
                .execute();
    }

    /**
     * Получить минус-гео для указанных баннеров.
     *
     * @param shard     номер шарда
     * @param bannerIds список id баннеров
     */
    public Map<Long, List<Long>> getBannersMinusGeo(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return Collections.emptyMap();
        }

        Field<String> concatenatedMinusGeoField = groupConcat(BANNERS_MINUS_GEO.MINUS_GEO).separator(",")
                .as(name("concatenatedMinusGeo"));

        Map<Long, String> pidToConcatenatedMinusGeoMap = dslContextProvider.ppc(shard)
                .select(BANNERS.BID, concatenatedMinusGeoField)
                .from(BANNERS)
                .join(BANNERS_MINUS_GEO).on(BANNERS_MINUS_GEO.BID.eq(BANNERS.BID))
                .where(BANNERS.BID.in(bannerIds))
                .groupBy(BANNERS.BID)
                .fetchMap(BANNERS.BID, concatenatedMinusGeoField);

        return EntryStream.of(pidToConcatenatedMinusGeoMap)
                .mapValues(this::bannerMinusGeoFromDB)
                .toMap();
    }


    /**
     * Получить минус-гео с типом current для всех активных баннеров из указанных групп.
     *
     * @param shard      номер шарда
     * @param adGroupIds список id групп
     * @return мапа id группы -> минус-гео этой группы
     */
    public Map<Long, List<Long>> getBannersMinusGeoByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return Collections.emptyMap();
        }

        Field<String> concatenatedMinusGeoField = groupConcat(BANNERS_MINUS_GEO.MINUS_GEO).separator(",")
                .as(name("concatenatedMinusGeo"));

        Map<Long, String> pidToConcatenatedMinusGeoMap = dslContextProvider.ppc(shard)
                .select(BANNERS.PID, concatenatedMinusGeoField)
                .from(BANNERS)
                .join(BANNERS_MINUS_GEO).on(BANNERS_MINUS_GEO.BID.eq(BANNERS.BID))
                .where(BANNERS.PID.in(adGroupIds))
                .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .and(BANNERS.STATUS_POST_MODERATE.ne(BannersStatuspostmoderate.Rejected))
                .and(BANNERS_MINUS_GEO.TYPE.eq(BannersMinusGeoType.current))
                .groupBy(BANNERS.PID)
                .fetchMap(BANNERS.PID, concatenatedMinusGeoField);

        return EntryStream.of(pidToConcatenatedMinusGeoMap)
                .mapValues(this::bannerMinusGeoFromDB)
                .toMap();
    }

    /**
     * Получить последние принятые от модерации (current) минус-гео для указанных баннеров.
     */
    public Map<Long, List<Long>> getCurrentBannersMinusGeo(int shard, Collection<Long> bannerIds) {
        return getCurrentBannersMinusGeo(dslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Получить последние принятые от модерации (current) минус-гео для указанных баннеров.
     */
    public Map<Long, List<Long>> getCurrentBannersMinusGeo(DSLContext contextProvider, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return Collections.emptyMap();
        }

        Map<Long, String> bannerToMinusGeo = contextProvider
                .select(BANNERS_MINUS_GEO.BID, BANNERS_MINUS_GEO.MINUS_GEO)
                .from(BANNERS_MINUS_GEO)
                .where(BANNERS_MINUS_GEO.BID.in(bannerIds))
                .and(BANNERS_MINUS_GEO.TYPE.eq(BannersMinusGeoType.current))
                .fetchMap(BANNERS.BID, BANNERS_MINUS_GEO.MINUS_GEO);

        return EntryStream.of(bannerToMinusGeo)
                .mapValues(this::bannerMinusGeoFromDB)
                .toMap();
    }


    public void deleteMinusGeo(int shard, Collection<Long> bannerIds) {
        deleteMinusGeo(dslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Удалить записи о минус-гео для баннеров
     * Выполняется только для минус-гео с типом current
     */
    public void deleteMinusGeo(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        context
                .delete(BANNERS_MINUS_GEO)
                .where(BANNERS_MINUS_GEO.BID.in(bannerIds))
                .and(BANNERS_MINUS_GEO.TYPE.eq(BannersMinusGeoType.current))
                .execute();
    }

    /**
     * Запросить перемодерацию баннеров (См. protected/Direct/Model/Manager.pm::_do_update_banners)
     */
    public void requestRemoderation(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard).transaction(config -> {
            config.dsl().update(BANNERS)
                    .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                    .set(BANNERS.STATUS_MODERATE, Ready)
                    .set(
                            BANNERS.STATUS_POST_MODERATE,
                            DSL.when(
                                    BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Rejected),
                                    BannersStatuspostmoderate.Rejected)
                                    .otherwise(BannersStatuspostmoderate.No))
                    .set(BANNERS.LAST_CHANGE, DSL.currentLocalDateTime())
                    .where(
                            BANNERS.BID.in(bannerIds)
                                    .and(BANNERS.STATUS_MODERATE.ne(New)))
                    .execute();

            deleteMinusGeo(config.dsl(), bannerIds);
        });
    }

    public Map<Long, Long> getBannerModerateVersions(DSLContext dslContext, Collection<Long> bannerIds) {
        return dslContext
                .select(BANNER_MODERATION_VERSIONS.BID, BANNER_MODERATION_VERSIONS.VERSION)
                .from(BANNER_MODERATION_VERSIONS)
                .where(BANNER_MODERATION_VERSIONS.BID.in(bannerIds))
                .fetchMap(BANNER_MODERATION_VERSIONS.BID, BANNER_MODERATION_VERSIONS.VERSION);
    }

    /**
     * Удалить флажки постмодерации и автомодерации (См. protected/Direct/Model/Manager
     * .pm::_do_clear_banners_moderation_flags)
     */
    public void clearModerationFlags(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard).transaction(config -> {
            config.dsl().delete(POST_MODERATE)
                    .where(POST_MODERATE.BID.in(bannerIds))
                    .execute();
            config.dsl().delete(AUTO_MODERATE)
                    .where(AUTO_MODERATE.BID.in(bannerIds))
                    .execute();
        });
    }

    private List<Long> bannerMinusGeoFromDB(String geo) {
        return Arrays.stream(geo.split(","))
                .filter(StringUtils::isNotEmpty)
                .map(Long::valueOf)
                .distinct()
                .sorted()
                .collect(toList());
    }

    public int updateStatusModerate(int shard, Collection<Long> bannerIds, BannerStatusModerate status) {
        return updateStatusModerate(dslContextProvider.ppc(shard), bannerIds, status, null);
    }

    public int updateStatusModerateIfNotDraft(int shard, Collection<Long> bannerIds, BannerStatusModerate status) {
        return updateStatusModerate(dslContextProvider.ppc(shard), bannerIds, status, BannerStatusModerate.NEW);
    }

    public int updateStatusModerate(DSLContext dsl, Collection<Long> bannerIds, BannerStatusModerate newStatus) {
        return updateStatusModerate(dsl, bannerIds, newStatus, null);
    }

    public int updateStatusModerate(DSLContext dsl, Collection<Long> bannerIds, BannerStatusModerate newStatus,
                                    @Nullable BannerStatusModerate oldStatusExclude) {
        if (bannerIds.isEmpty()) {
            return 0;
        }
        var query = dsl
                .update(BANNERS)
                .set(BANNERS.STATUS_MODERATE, BannerStatusModerate.toSource(newStatus))
                .where(BANNERS.BID.in(bannerIds));
        if (null != oldStatusExclude) {
            query = query.and(BANNERS.STATUS_MODERATE.ne(BannerStatusModerate.toSource(oldStatusExclude)));
        }
        return query.execute();
    }

    public void updateStatusModerateAndPostModerate(int shard, Collection<Long> bannerIds,
                                                    BannerStatusModerate status,
                                                    BannersStatuspostmoderate statusPostModerate) {
        updateStatusModerateAndPostModerate(dslContextProvider.ppc(shard).configuration(),
                bannerIds, status, statusPostModerate);
    }

    public void updateStatusModerateAndPostModerate(Configuration cfg, Collection<Long> bannerIds,
                                                    BannerStatusModerate status,
                                                    BannersStatuspostmoderate statusPostModerate) {
        if (bannerIds.isEmpty()) {
            return;
        }

        DSL.using(cfg)
                .update(BANNERS)
                .set(BANNERS.STATUS_MODERATE, BannerStatusModerate.toSource(status))
                .set(BANNERS.STATUS_POST_MODERATE, statusPostModerate)
                .where(BANNERS.BID.in(bannerIds))
                .execute();
    }

    public void updateModerateBannerPages(Configuration configuration, long moderateBannerPageId,
                                          @Nullable String comment,
                                          @Nullable String taskUrl, StatusModerateBannerPage status) {
        DSL.using(configuration)
                .update(MODERATE_BANNER_PAGES)
                .set(MODERATE_BANNER_PAGES.STATUS_MODERATE, StatusModerateBannerPage.toSource(status))
                .set(MODERATE_BANNER_PAGES.COMMENT, comment)
                .set(MODERATE_BANNER_PAGES.TASK_URL, taskUrl)
                .where(MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID.eq(moderateBannerPageId))
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .execute();
    }

    /**
     * Получаем список геолицензируемых флагов на неархивных баннерах кампании
     *
     * @param shard
     * @param campaignIds
     * @return
     */
    public Map<Long, Set<String>> getBannerGeoLegalFlagsByCids(int shard, Collection<Long> campaignIds) {
        if (campaignIds == null) {
            return emptyMap();
        }

        HashSet<String> bannerGeoLegalFlags = new HashSet<>();
        for (BannerGeoLegalFlags c : BannerGeoLegalFlags.values()) {
            bannerGeoLegalFlags.add(c.getValue());
        }

        Map<Long, Set<String>> result = new HashMap<>();

        dslContextProvider.ppc(shard)
                .select(BANNERS.CID, BANNERS.FLAGS)
                .from(BANNERS).join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .where(BANNERS.CID.in(campaignIds))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .and(BANNERS.FLAGS.isNotNull())
                .fetch()
                .forEach(r -> {
                    Long campaignId = r.get(BANNERS.CID);
                    BannerFlags flags = BannerFlags.fromSource(r.get(BANNERS.FLAGS));
                    if (flags == null) {
                        return;
                    }
                    for (String flag : flags.getFlags().keySet()) {
                        if (bannerGeoLegalFlags.contains(flag)) {
                            Set<String> campFlags = result.getOrDefault(campaignId, new HashSet<>());
                            campFlags.add(flag);
                            result.put(campaignId, campFlags);
                        }
                    }
                });
        return result;
    }

    /**
     * Возвращает статус архивности баннеров с учётом архивности кампании
     * (если кампания заархивирована, любой баннер в ней тоже будет считаться архивным)
     */
    public Map<Long, Boolean> getBannersArchiveStatus(int shard, Collection<Long> bannerIds) {
        if (bannerIds == null || bannerIds.isEmpty()) {
            return emptyMap();
        }
        Map<Long, Boolean> result = new HashMap<>();
        dslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.STATUS_ARCH, CAMPAIGNS.ARCHIVED)
                .from(BANNERS).join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .where(BANNERS.BID.in(bannerIds))
                .fetch()
                .forEach(r -> {
                    Long bannerId = r.get(BANNERS.BID);
                    BannersStatusarch statusArch = r.get(BANNERS.STATUS_ARCH);
                    CampaignsArchived campaignArchived = r.get(CAMPAIGNS.ARCHIVED);
                    boolean bannerArchived = statusArch == BannersStatusarch.Yes
                            || campaignArchived == CampaignsArchived.Yes;
                    result.put(bannerId, bannerArchived);
                });
        return result;
    }

    /**
     * Получает по списку cid все уникальные домены и флаги для баннеров этих cid
     */
    public Map<CidAndDomain,Set<String>> getCidAndBannersInfo(int shard, Collection<Long> campaignIds) {
        return StreamEx.of(
                dslContextProvider.ppc(shard)
                        .select(BANNERS.CID, BANNERS.REVERSE_DOMAIN, FLAGS_WITHOUT_BLANKS)
                        .from(BANNERS)
                        .where(BANNERS.CID.in(campaignIds))
                        .and(BANNERS.REVERSE_DOMAIN.isNotNull())
                        .groupBy(BANNERS.CID, BANNERS.REVERSE_DOMAIN, FLAGS_WITHOUT_BLANKS)
                        .fetch())
                .groupingBy(r -> CidAndDomain.of(r.get(BANNERS.CID), r.get(BANNERS.REVERSE_DOMAIN)),
                        flatMapping(r -> r.get(FLAGS_WITHOUT_BLANKS) == null
                                ? Stream.empty()
                                : Arrays.stream(r.get(FLAGS_WITHOUT_BLANKS).split(",")), toSet()));
    }

    /**
     * Получает по списку cid все минус гео и домены для баннеров этих cid
     */
    public Map<CidAndDomain, String> getDomainAndMinusGeoByCampaignIds(int shard,
                                                                       List<Long> campaignIds) {
        return StreamEx.of(
                dslContextProvider.ppc(shard)
                        .selectDistinct(BANNERS.CID, BANNERS.REVERSE_DOMAIN, CONCATENATED_MINUS_GEO_WIth_TYPE)
                        .from(BANNERS)
                        .join(BANNERS_MINUS_GEO).on(BANNERS.BID.eq(BANNERS_MINUS_GEO.BID))
                        .where(BANNERS.CID.in(campaignIds))
                        .and(BANNERS.REVERSE_DOMAIN.isNotNull())
                        .fetch())
                .groupingBy(r -> CidAndDomain.of(r.get(BANNERS.CID), r.get(BANNERS.REVERSE_DOMAIN)),
                        mapping(r -> r.get(CONCATENATED_MINUS_GEO_WIth_TYPE), joining(" ")));
    }

    public Map<Long, BannersStatusmoderate> getBannersStatusModerate(Configuration conf, Collection<Long> bannerIds) {
        return conf.dsl()
                .select(BANNERS.BID, BANNERS.STATUS_MODERATE)
                .from(BANNERS)
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(BANNERS.BID, BANNERS.STATUS_MODERATE);
    }
}
