package ru.yandex.direct.core.entity.banner.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.MultimapBuilder;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.AggregateFunction;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.Select;
import org.jooq.SelectHavingStep;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.BannerAdGroupRelation;
import ru.yandex.direct.core.entity.banner.container.AdsCountCriteria;
import ru.yandex.direct.core.entity.banner.container.AdsSelectionCriteria;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BidGroup;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.type.BannerRepositoryTypeSupportFacade;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersPhoneflag;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.OrganizationsStatusPublish;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.castNull;
import static org.jooq.impl.DSL.coalesce;
import static org.jooq.impl.DSL.count;
import static org.jooq.impl.DSL.ifnull;
import static org.jooq.impl.DSL.max;
import static org.jooq.impl.DSL.min;
import static org.jooq.impl.DSL.noCondition;
import static org.jooq.impl.DSL.when;
import static ru.yandex.direct.core.entity.banner.repository.BannerRepositoryConstants.BANNER_CLASS_TO_TYPE;
import static ru.yandex.direct.core.entity.banner.repository.selection.Converter.buildSelectConditionStep;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PUBLISHER;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.ORGANIZATIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate.Yes;
import static ru.yandex.direct.dbschema.ppc.tables.BannerDisplayHrefs.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class BannerRelationsRepository implements BannerAdGroupRelation {

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final BannerRepositoryTypeSupportFacade bannerRepositoryTypeSupportFacade;

    @Autowired
    public BannerRelationsRepository(DslContextProvider dslContextProvider,
                                     ShardHelper shardHelper,
                                     BannerRepositoryTypeSupportFacade bannerRepositoryTypeSupportFacade) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.bannerRepositoryTypeSupportFacade = bannerRepositoryTypeSupportFacade;
    }

    public Map<Long, Long> getCampaignIdsByBannerIdsForShard(int shard, Collection<Long> bids) {
        return getCampaignIdsByBannerIds(dslContextProvider.ppc(shard), bids);
    }

    /**
     * Получает map id банера -> id кампании по списку id баннеров и типу баннеров.
     * При походе в БД чанкуем {@code bannerIds} пачками по {@value SqlUtils#TYPICAL_SELECT_CHUNK_SIZE} штук
     *
     * @param bannerIds список id баннеров
     * @return map id банера -> id кампании
     */
    public Map<Long, Long> getCampaignIdsByBannerIdsAndBannerType(int shard, Collection<Long> bannerIds,
                                                                   Class<? extends Banner> bannerClass) {
        BannersBannerType type = BANNER_CLASS_TO_TYPE.get(bannerClass);
        Map<Long, Long> result = new HashMap<>();
        Iterables.partition(bannerIds, SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .forEach(chunk -> {
                    Map<Long, Long> chunkResult = dslContextProvider.ppc(shard)
                            .select(BANNERS.BID, BANNERS.CID)
                            .from(BANNERS)
                            .where(BANNERS.BID.in(chunk).and(BANNERS.BANNER_TYPE.eq(type)))
                            .fetchMap(BANNERS.BID, BANNERS.CID);
                    result.putAll(chunkResult);
                });
        return result;
    }

    /**
     * Получает map id банера -> id кампании по списку id баннеров.
     * При походе в БД чанкуем {@code bannerIds} пачками по {@value SqlUtils#TYPICAL_SELECT_CHUNK_SIZE} штук
     *
     * @param bannerIds список id баннеров
     * @return map id банера -> id кампании
     */
    public Map<Long, Long> getCampaignIdsByBannerIds(DSLContext context, Collection<Long> bannerIds) {
        Map<Long, Long> result = new HashMap<>();
        Iterables.partition(bannerIds, SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .forEach(chunk -> {
                    Map<Long, Long> chunkResult = context
                            .select(BANNERS.BID, BANNERS.CID)
                            .from(BANNERS)
                            .where(BANNERS.BID.in(chunk))
                            .fetchMap(BANNERS.BID, BANNERS.CID);
                    result.putAll(chunkResult);
                });
        return result;
    }

    /**
     * Получает map id банера -> id кампании по списку id баннеров
     *
     * @param bannerIds список id баннеров
     * @return map id банера -> id кампании
     */
    public Map<Long, Long> getCampaignIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        DSLContext context = dslContextProvider.ppc(shard);
        return getCampaignIdsByBannerIds(context, bannerIds);
    }

    /**
     * Получает все баннеры кампаний из {@code campaignIds}
     *
     * @return map[bid -> cid]
     */
    public Map<Long, Long> getBannerIdsMapByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.CID)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds))
                .fetchMap(BANNERS.BID, BANNERS.CID);
    }

    /**
     * Получает все баннеры кампаний из {@code campaignIds}, тип которых содержится в {@code bannerClasses}
     *
     * @return map[bid -> cid]
     */
    public Map<Long, Long> getBannerIdsMapByCampaignIdsAndBannerTypes(int shard, Collection<Long> campaignIds,
                                                                      Collection<Class<? extends Banner>> bannerClasses) {
        @Nullable Condition condition = bannerRepositoryTypeSupportFacade.getConditionThatRecordAnyOfClasses(bannerClasses);
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.CID)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds)
                        .and(nvl(condition, noCondition())))
                .fetchMap(BANNERS.BID, BANNERS.CID);
    }

    /**
     * Получает коллекцию bannerIds по коллекции id групп {@param adGroupIds}
     */
    public List<Long> getBannerIdsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds))
                .fetch(BANNERS.BID);
    }

    /**
     * Получает коллекцию bannerIds по коллекции id кампаний {@param campaignIds}
     */
    public List<Long> getBannerIdsByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds))
                .fetch(BANNERS.BID);
    }

    public Map<Long, List<Long>> getBannerIdsByCampaignIdsMap(int shard, Collection<Long> campaignIds) {
        return getBannerIdsByCampaignIdsMap(dslContextProvider.ppc(shard), campaignIds);
    }

    public Map<Long, List<Long>> getBannerIdsByCampaignIdsMap(DSLContext context, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyMap();
        }

        return context
                .select(BANNERS.CID, BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds))
                .fetchGroups(BANNERS.CID, BANNERS.BID);
    }

    public List<Long> getNonArchivedBannerIds(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyList();
        }

        return dslContextProvider.ppc(shard)
                .select(BANNERS.CID, BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .fetch(BANNERS.BID);
    }

    /**
     * Получает все не архивные баннеры кампаний из {@code campaignIds}
     *
     * @return map[bid -> cid]
     */
    public Map<Long, Long> getCampaignIdByNonArchivedBannerId(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(BANNERS.CID, BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .fetchMap(BANNERS.BID, BANNERS.CID);
    }

    /**
     * @param adGroupIds список id групп
     * @return map id группа -> список id баннеров
     */
    public Multimap<Long, Long> getAdGroupIdToBannerIds(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return MultimapBuilder.hashKeys().hashSetValues().build();
        }

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(BANNERS.BID, BANNERS.PID))
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds))
                .fetch();
        Multimap<Long, Long> multimap = MultimapBuilder.hashKeys().hashSetValues(50).build();
        result.forEach(record -> multimap.put(record.get(BANNERS.PID), record.get(BANNERS.BID)));
        return multimap;
    }

    public Map<Long, Long> getBannerIdsToSitelinkSetIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.SITELINKS_SET_ID)
                .from(BANNERS)
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(BANNERS.BID, BANNERS.SITELINKS_SET_ID);
    }

    /**
     * Возвращает map id баннера -> тип группы
     *
     * @param shard     шард
     * @param bannerIds идентификаторы баннеров
     * @return map id баннера -> тип группы
     */
    public Map<Long, PhrasesAdgroupType> getAdGroupTypesByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, PHRASES.ADGROUP_TYPE)
                .from(BANNERS)
                .innerJoin(PHRASES)
                .on(BANNERS.PID.eq(PHRASES.PID))
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(BANNERS.BID, PHRASES.ADGROUP_TYPE);
    }

    /**
     * Возвращает map id группы -> список id неархивных баннеров
     *
     * @param shard         шард
     * @param campaignId    идентификатор кампании
     * @param bannerClasses классы баннеров для поиска
     * @return map id группы -> список id неархивных баннеров
     */
    public Map<Long, List<Long>> getAdGroupIdsToNonArchivedBannerIds(
            int shard, Long campaignId, Collection<Class<? extends Banner>> bannerClasses) {
        List<BannersBannerType> types = mapList(bannerClasses, BANNER_CLASS_TO_TYPE::get);
        return dslContextProvider.ppc(shard)
                .select(PHRASES.PID, BANNERS.BID)
                .from(PHRASES)
                .leftJoin(BANNERS).on(PHRASES.PID.eq(BANNERS.PID))
                .where(PHRASES.CID.eq(campaignId)
                        .and(BANNERS.BANNER_TYPE.in(types))
                        .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)))
                .fetchGroups(PHRASES.PID, BANNERS.BID);
    }

    /**
     * Возвращает map id баннера -> id кампании
     * Выбирает для каждой запрошенной кампании неархивные баннеры выбранных типов, которые редактировались последними
     * В случае совпадения LastChange у двух или более баннеров одного типа, выбирается тот, у которого наибольший bid
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний, для которых надо найти самые "свежие" баннеры
     * @param bannerTypes типы баннеров для поиска
     * @return map id баннера -> id кампании
     */
    public Map<Long, Long> getLastChangedBannerIdsWithCampaignIds(int shard, Collection<Long> campaignIds,
                                                                  Collection<BannersBannerType> bannerTypes) {

        return dslContextProvider.ppc(shard)
                .select(max(BANNERS.BID).as(BANNERS.BID), BANNERS.CID, BANNERS.BANNER_TYPE,
                        BANNERS.LAST_CHANGE)
                .from(BANNERS)
                .innerJoin(DSL.select(BANNERS.CID, BANNERS.BANNER_TYPE,
                        max(BANNERS.LAST_CHANGE).as(BANNERS.LAST_CHANGE))
                        .from(BANNERS)
                        .where(BANNERS.CID.in(campaignIds).and(BANNERS.BANNER_TYPE.in(bannerTypes))
                                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)))
                        .groupBy(BANNERS.CID, BANNERS.BANNER_TYPE))
                .using(BANNERS.CID, BANNERS.BANNER_TYPE, BANNERS.LAST_CHANGE)
                .groupBy(BANNERS.CID, BANNERS.BANNER_TYPE, BANNERS.LAST_CHANGE)
                .fetchMap(BANNERS.BID, BANNERS.CID);
    }

    /**
     * Получает коллекцию bannerIds по коллекции id кампаний {@param campaignIds}
     * и коллекции типов баннера {@param bannerClasses}
     */
    public List<Long> getBannerIdsByCampaignIdsAndBannerTypes(int shard, Collection<Long> campaignIds,
                                                              Collection<Class<? extends Banner>> bannerClasses) {
        List<BannersBannerType> types = mapList(bannerClasses, BANNER_CLASS_TO_TYPE::get);
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds))
                .and(BANNERS.BANNER_TYPE.in(types))
                .fetch(BANNERS.BID);
    }

    /**
     * @see #getAdGroupIdsByBannerIds(DSLContext, Collection)
     */
    public Map<Long, Long> getAdGroupIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        return getAdGroupIdsByBannerIds(dslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Получает map id банера -> id группы по списку id баннеров
     *
     * @param bannerIds список id баннеров
     * @return map id банера -> id группы
     */
    public Map<Long, Long> getAdGroupIdsByBannerIds(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        return context
                .select(BANNERS.BID, BANNERS.PID)
                .from(BANNERS)
                .where(BANNERS.BID.in(bannerIds))
                .fetchMap(BANNERS.BID, BANNERS.PID);
    }

    public Set<Long> bannersWithDeclinedAdGroups(Configuration configuration, Collection<Long> bids) {
        return configuration.dsl()
                .selectDistinct(BANNERS.BID)
                .from(BANNERS.join(PHRASES).on(BANNERS.PID.eq(PHRASES.PID)))
                .where(PHRASES.STATUS_MODERATE.eq(PhrasesStatusmoderate.No),
                        BANNERS.BID.in(bids))
                .fetchSet(Record1::value1);
    }

    public Set<Long> getHasRunningUnmoderatedBannersByCampaignId(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.CID)
                .from(PHRASES)
                .join(BANNERS).on(BANNERS.PID.equal(PHRASES.PID))
                .leftJoin(BANNER_PERMALINKS).on(BANNER_PERMALINKS.BID.equal(BANNERS.BID))
                .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.equal(BannerPermalinksPermalinkAssignType.manual))
                .and(BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK.equal(0L))
                .leftJoin(ORGANIZATIONS).on(ORGANIZATIONS.PERMALINK_ID.equal(BANNER_PERMALINKS.PERMALINK))
                .and(ORGANIZATIONS.STATUS_PUBLISH.equal(OrganizationsStatusPublish.published))
                .where(PHRASES.CID.in(campaignIds)
                        .and(BANNERS.STATUS_ACTIVE.equal(BannersStatusactive.Yes))
                        .and(BANNERS.STATUS_SHOW.equal(BannersStatusshow.Yes))
                        .andNot(PHRASES.STATUS_POST_MODERATE.equal(PhrasesStatuspostmoderate.Yes)
                                .and(BANNERS.STATUS_POST_MODERATE.equal(BannersStatuspostmoderate.Yes))
                                .and(ifnull(BANNERS.HREF, "").notEqual("")
                                        .or(BANNERS.PHONEFLAG.equal(BannersPhoneflag.Yes))
                                        .or(BANNERS.BANNER_TYPE.equal(BannersBannerType.dynamic))
                                        .or(BANNER_PERMALINKS.PERMALINK.isNotNull()
                                                .and(ORGANIZATIONS.PERMALINK_ID.isNotNull())
                                        )
                                )
                        )
                        .and(PHRASES.STATUS_MODERATE.notEqual(PhrasesStatusmoderate.Yes)
                                .or(BANNERS.STATUS_MODERATE.notEqual(Yes))
                                .or(ifnull(BANNERS.HREF, "").equal("")
                                        .and(BANNERS.BANNER_TYPE.equal(BannersBannerType.text))
                                        .and(BANNERS.PHONEFLAG.notEqual(BannersPhoneflag.Yes))
                                        .and(BANNER_PERMALINKS.PERMALINK.isNull()
                                                .or(ORGANIZATIONS.PERMALINK_ID.isNull())
                                        )
                                )
                        )
                )
                .fetchSet(BANNERS.CID);
    }

    /**
     * Получить главный баннер по каждой из групп, чьи ID переданы в {@code adGroupIds}.
     * <p>
     * Главный баннер тут -- это первый неграфический при сортировке по
     * {@link TextBanner#ID}
     */
    public Map<Long, BannerWithSystemFields> getMainBannerByAdGroupIds(ClientId clientId,
                                                                       Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getMainBannerByAdGroupIds(shard, adGroupIds);
    }

    /**
     * Получает список "главных" баннеров по списку id групп.
     * <p>
     * "Главный" баннер - это баннер, на который будут расчитываться ставки в Торгах. Если в группе есть текстовые
     * ({@code banner_type != 'image_ad'}) баннеры, то главный тот из них, у кого минимальный ID. Если все баннеры
     * картиночные,
     * то возвращается баннер с минимальным ID из них.
     * !!Уточнения не возвращаются данным методом
     *
     * @param shard      шард
     * @param adGroupIds список id групп извлекаемых баннеров
     * @return список баннеров
     */
    @QueryWithoutIndex("Нужно допиливать ExplainListener, индекс там где-то внутри есть")
    public Map<Long, BannerWithSystemFields> getMainBannerByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        String mainBannerIdField = "mainBannerId";
        // активный текстовый баннер с минимальным ID в группе
        AggregateFunction<Long> minActiveTextBannerId = min(when(BANNERS.BANNER_TYPE
                        .ne(BannersBannerType.image_ad)
                        .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                        .and(BANNERS.STATUS_ACTIVE.eq(BannersStatusactive.Yes)
                                .or(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes)
                                        .and(PHRASES.STATUS_POST_MODERATE.eq(PhrasesStatuspostmoderate.Yes))
                                        .and(BANNERS.HREF.isNotNull().or(BANNERS.PHONEFLAG.eq(BannersPhoneflag.Yes))))),
                BANNERS.BID).otherwise(castNull(BANNERS.BID)));

        // текстовый баннер с минимальным ID в группе
        AggregateFunction<Long> minTextBannerId =
                min(when(BANNERS.BANNER_TYPE.ne(BannersBannerType.image_ad), BANNERS.BID)
                        .otherwise(castNull(BANNERS.BID)));

        // картиночный баннер (используется, если не нашёлся текстовый)
        AggregateFunction<Long> minImageBannerId =
                min(when(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad), BANNERS.BID)
                        .otherwise(castNull(BANNERS.BID)));

        Select<Record1<Long>> mainBannersSubquery = dslContextProvider.ppc(shard)
                .select(coalesce(minActiveTextBannerId, minTextBannerId, minImageBannerId)
                        .as(mainBannerIdField))
                .from(BANNERS)
                .leftJoin(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .where(BANNERS.PID.in(adGroupIds))
                .groupBy(BANNERS.PID);

        Result<Record> result = bannerRepositoryTypeSupportFacade.collectSelectJoinStep(
                dslContextProvider.ppc(shard)
                        .select(bannerRepositoryTypeSupportFacade.getAllModelFields())
                        .from(mainBannersSubquery)
                        .join(BANNERS).on(BANNERS.BID.eq(mainBannersSubquery.field(mainBannerIdField, Long.class)))
                        .getQuery())
                .fetch();

        Map<Long, BannerWithSystemFields> mainBannerByAdGroupIds = StreamEx.of(result)
                .map(bannerRepositoryTypeSupportFacade::getModelFromRecord)
                .select(BannerWithSystemFields.class)
                .mapToEntry(BannerWithSystemFields::getAdGroupId)
                .invert()
                .toMap();

        bannerRepositoryTypeSupportFacade.enrichModelFromOtherTables(dslContextProvider.ppc(shard),
                new ArrayList<>(mainBannerByAdGroupIds.values()));

        return mainBannerByAdGroupIds;
    }

    /**
     * Получает отображение id кампании в список id банеров по заданным условиям отбора
     *
     * @param shard             шард
     * @param selectionCriteria условия отбора
     * @return отображение id кампании в список id банеров
     */
    public Map<Long, List<Long>> getCampaignIdToBannerIdsBySelectionCriteria(int shard,
                                                                             AdsSelectionCriteria selectionCriteria) {
        SelectJoinStep<Record2<Long, Long>> selectJoinStep =
                dslContextProvider.ppc(shard).select(BANNERS.CID, BANNERS.BID).from(BANNERS);

        SelectHavingStep<Record2<Long, Long>> selectConditionStep =
                buildSelectConditionStep(selectJoinStep, selectionCriteria);

        // TODO: нужно ли делать аналог API::Service::Ads::_fix_where_for_images?

        return selectConditionStep
                .orderBy(BANNERS.BID)
                .fetchGroups(BANNERS.CID, BANNERS.BID);
    }

    public Set<BidGroup> getBidGroups(Integer shard, Collection<String> reversedDomains,
                                      Collection<Long> clientUserIds) {

        List<Condition> reversedDomainCondition = new ArrayList<>();

        for (String reversedDomain : reversedDomains) {
            reversedDomainCondition.add(BANNERS.REVERSE_DOMAIN.like(reversedDomain + ".%"));
            reversedDomainCondition.add(BANNERS.REVERSE_DOMAIN.eq(reversedDomain));
        }

        Map<BidGroup, List<Long>> result = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, BANNERS.DOMAIN, BANNERS.BID, USERS.FIO, USERS.UID)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .join(USERS).on(USERS.UID.eq(CAMPAIGNS.UID))
                .where(CAMPAIGNS.UID.in(clientUserIds))
                .and(DSL.or(reversedDomainCondition))
                .fetchGroups(r -> new BidGroup()
                                .withCampaignId(r.get(CAMPAIGNS.CID))
                                .withDomain(r.get(BANNERS.DOMAIN))
                                .withFio(r.get(USERS.FIO))
                                .withClientUserId(r.get(USERS.UID)),
                        r -> r.get(BANNERS.BID));
        result.forEach(BidGroup::setBids);
        return result.keySet();
    }

    public Map<Long, ClientId> getClientIdsByBannerIds(Set<Long> bannerIds) {
        Map<Long, ClientId> result = new HashMap<>();
        shardHelper.groupByShard(bannerIds, ShardKey.BID)
                .forEach((k, v) -> result.putAll(getClientIdsByBannerIds(k, v)));
        return result;
    }

    public Map<Long, ClientId> getClientIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, CAMPAIGNS.CLIENT_ID)
                .from(BANNERS)
                .leftJoin(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.BID.in(bannerIds))
                .and(CAMPAIGNS.CLIENT_ID.isNotNull())
                .fetchMap(BANNERS.BID, r -> ClientId.fromLong(r.get(CAMPAIGNS.CLIENT_ID)));
    }

    /**
     * Возвращает ID главных баннеров для групп
     *
     * @param clientId   идентификатор клиента
     * @param adGroupIds идентификаторы групп объявлений
     * @return {@link Map&lt;Long, Long&gt;} с соответствием ID группы и ID главного объявления
     */
    public Map<Long, Long> getMainBannerIdsByAdGroupIds(ClientId clientId, Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return getMainBannerIdsByAdGroupIds(shard, adGroupIds);
    }

    /**
     * Возвращает соответствие ID групп ID главных баннеров.
     * <p>
     * Главный баннер &ndash; это показываемый баннер ({@code statusShow='Yes'}), который при этом
     * либо активный ({@code statusActive='Yes'}),
     * либо готов стать таким ({@code statusPostModerate='Yes'} у баннера и группы,
     * и задана ссылка {@code banners.href} или телефон {@code banners.phoneflag='Yes'}).
     * <p>
     * Если по указанным параметрам ничего не находится, главным считается первый по {@code bid}.
     * <p>
     * Если группа не существует, то её ID не будет в результирующей {@link Map}'е.
     * <p>
     * Референсная реализация в Perl: {@code Primitives::get_main_banner_ids_by_pids}
     */
    public Map<Long, Long> getMainBannerIdsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        Result<Record2<Long, Long>> activeMainBannersResult = dslContextProvider.ppc(shard)
                .select(BANNERS.PID, DSL.min(BANNERS.BID).as(BANNERS.BID))
                .hint("STRAIGHT_JOIN")
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .where(BANNERS.PID.in(adGroupIds))
                .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                .and(BANNERS.STATUS_ACTIVE.eq(BannersStatusactive.Yes)
                        .or(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes)
                                .and(PHRASES.STATUS_POST_MODERATE.eq(PhrasesStatuspostmoderate.Yes))
                                .and(DSL.ifnull(BANNERS.HREF, "").ne("")
                                        .or(BANNERS.PHONEFLAG.eq(BannersPhoneflag.Yes))
                                )
                        )
                )
                .groupBy(BANNERS.PID)
                .fetch();
        Map<Long, Long> activeMainBannersByAdGroupId = StreamEx.of(activeMainBannersResult)
                .mapToEntry(Record2::value1, Record2::value2)
                .toMap();
        List<Long> adGroupsWithoutActiveBanner = StreamEx.of(adGroupIds)
                .remove(activeMainBannersByAdGroupId::containsKey)
                .toList();
        Result<Record2<Long, Long>> inactiveMainBanners = dslContextProvider.ppc(shard)
                .select(BANNERS.PID, DSL.min(BANNERS.BID).as(BANNERS.BID))
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupsWithoutActiveBanner))
                .groupBy(BANNERS.PID)
                .fetch();
        return StreamEx.of(inactiveMainBanners)
                .mapToEntry(Record2::value1, Record2::value2)
                .append(activeMainBannersByAdGroupId)
                .toMap();
    }

    public Set<Long> getTemplateBannerIdsByAdGroupIds(
            int shard, ClientId clientId, Collection<Long> adGroupIds) {
        String pattern = "%#%#%";
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .leftJoin(BANNER_DISPLAY_HREFS).on(BANNER_DISPLAY_HREFS.BID.eq(BANNERS.BID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.PID.in(adGroupIds))
                .and(BANNERS.TITLE.like(pattern)
                        .or(BANNERS.TITLE_EXTENSION.like(pattern))
                        .or(BANNERS.BODY.like(pattern))
                        .or(BANNERS.HREF.like(pattern))
                        .or(BANNER_DISPLAY_HREFS.DISPLAY_HREF.like(pattern)))
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .fetchSet(BANNERS.BID);
    }

    /**
     * Проверяет, принадлежит ли баннер клиенту
     *
     * @param shard    шард
     * @param bid      id баннера
     * @param clientId клиент
     * @return true, если баннер принаждежит кленту,
     * false, если не принадлежит
     */
    public boolean bannerBelongsToClient(int shard, long bid, ClientId clientId) {
        int bannersCount = dslContextProvider.ppc(shard)
                .selectCount()
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.BID.eq(bid)
                        .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetchOne()
                .value1();
        return bannersCount != 0;
    }

    /**
     * Возвращает мапу существующих соответствий clientId к bannerIds
     * (массовый аналог bannerBelongsToClient)
     *
     * @param shard шард
     * @param bids  список bannerIds
     * @return Мапа существующих идентификаторов. Key - clientId, value - сет bannerIds
     */
    public Map<Long, Set<Long>> getExistingClientIdToBids(int shard, Set<Long> bids) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CLIENT_ID, BANNERS.BID)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.BID.in(bids))
                .fetchGroups(CAMPAIGNS.CLIENT_ID, BANNERS.BID)
                .entrySet()
                .stream()
                .collect(toMap(Map.Entry::getKey, y -> new HashSet<>(y.getValue())));
    }

    /**
     * Получение кол-ва существующих баннеров по кампаниям
     *
     * @param shard        шард
     * @param campaignsIds id запрашиваемых кампаний
     * @return соответствие "id кампании - количество баннеров в кампании"
     */
    public Map<Long, Integer> getBannersCounterByCampaigns(int shard, Collection<Long> campaignsIds) {
        return getBannersCounterByCampaigns(shard, campaignsIds, null);
    }

    /**
     * Получение кол-ва существующих баннеров по кампаниям
     *
     * @param shard         шард
     * @param campaignsIds  id запрашиваемых кампаний
     * @param countCriteria дополнительный критерий, какие баннеры учитывать
     * @return соответствие "id кампании - количество баннеров в кампании"
     */
    public Map<Long, Integer> getBannersCounterByCampaigns(int shard, Collection<Long> campaignsIds,
                                                           @Nullable AdsCountCriteria countCriteria) {
        String bidCounterAlias = "BID_COUNTER";

        return dslContextProvider.ppc(shard)
                .select(asList(BANNERS.CID, count(BANNERS.BID).as(bidCounterAlias)))
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignsIds))
                .and(getCondition(countCriteria))
                .groupBy(BANNERS.CID)
                .fetchMap(r -> r.getValue(BANNERS.CID), r -> r.getValue(bidCounterAlias, Integer.class));
    }

    /**
     * Получение кол-ва существующих и не архивных баннеров по группам
     *
     * @param shard      шард
     * @param adGroupIds id запрашиваемых групп
     * @return соответствие "id группы - <количество баннеров в группе, количество не архивных баннеров в группе>"
     */
    public Map<Long, Pair<Long, Long>> getBannersCountersByAdgroups(int shard, Collection<Long> adGroupIds) {
        String bidCounterAlias = "BID_COUNTER";

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(BANNERS.PID, BANNERS.STATUS_ARCH, DSL.count().as(bidCounterAlias)))
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds))
                .groupBy(BANNERS.PID, BANNERS.STATUS_ARCH)
                .fetch();

        Map<Long, Pair<Long, Long>> adGroupIdToBannersCounters = new HashMap<>();
        result.forEach(r -> {
            Long adGroupId = r.getValue(BANNERS.PID);
            Long bannersCount = r.getValue(bidCounterAlias, Long.class);
            BannersStatusarch statusArch = r.get(BANNERS.STATUS_ARCH);
            boolean bannerArchived = BannersStatusarch.Yes.equals(statusArch);

            Pair<Long, Long> allAndNotArchived = adGroupIdToBannersCounters.get(adGroupId);
            long allBannersCount = bannersCount + (allAndNotArchived != null ? allAndNotArchived.getLeft() : 0L);
            long notArchivedBannersCount = allAndNotArchived != null ? allAndNotArchived.getRight() : 0L;
            if (!bannerArchived) {
                notArchivedBannersCount += bannersCount;
            }

            adGroupIdToBannersCounters.put(adGroupId, Pair.of(allBannersCount, notArchivedBannersCount));
        });

        return adGroupIdToBannersCounters;
    }

    /**
     * По списку ID кампаний получить мапу ID кампании -> список ID баннеров, где в баннерах были изменения с
     * указанного времени.
     *
     * @param shard        шард
     * @param campaignIds  список id кампаний извлекаемых баннеров
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения в баннерах
     * @param limit        максимальное количество данных в ответе
     */
    public Map<Long, List<Long>> getChangedBannersByCampaignIds(
            int shard,
            Collection<Long> campaignIds,
            LocalDateTime fromDateTime,
            int limit
    ) {
        return dslContextProvider.ppc(shard)
                .select(Tables.BANNERS.CID, Tables.BANNERS.BID)
                .from(Tables.BANNERS)
                .leftJoin(BANNER_IMAGES).using(Tables.BANNERS.BID)
                .where(Tables.BANNERS.CID.in(campaignIds))
                .and(Tables.BANNERS.LAST_CHANGE.ge(fromDateTime)
                        .or(BANNER_IMAGES.DATE_ADDED.ge(fromDateTime))
                )
                .orderBy(Tables.BANNERS.CID)
                .limit(limit)
                .fetchGroups(Tables.BANNERS.CID, Tables.BANNERS.BID);
    }

    /**
     * Выбирать из списка баннеров такие, в которых произошли изменения начиная с указанного момента времени.
     *
     * @param shard        шард
     * @param bidIds       список id баннеров
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения в баннерах
     */
    public List<Long> getChangedBannersByIds(
            int shard,
            Collection<Long> bidIds,
            LocalDateTime fromDateTime
    ) {
        return dslContextProvider.ppc(shard)
                .select(Tables.BANNERS.BID)
                .from(Tables.BANNERS)
                .where(Tables.BANNERS.BID.in(bidIds))
                .and(Tables.BANNERS.LAST_CHANGE.ge(fromDateTime))
                .fetch(Tables.BANNERS.BID);
    }

    /**
     * Возвращает количество баннеров для каждой групп, в которых есть баннеры. Если баннеров нет,
     * в результате будет отсутствовать соответствующий {@code adGroupId}.
     * <p>
     * Note: статусы баннеров не учитываются.
     */
    public Map<Long, Integer> getBannerQuantitiesByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return getBannerQuantitiesByAdGroupIds(shard, adGroupIds, null);
    }

    /**
     * Возвращает количество баннеров для каждой групп, в которых есть баннеры. Если баннеров нет,
     * в результате будет отсутствовать соответствующий {@code adGroupId}.
     * <p>
     * Note: статусы баннеров не учитываются.
     */
    public Map<Long, Integer> getBannerQuantitiesByAdGroupIds(int shard, Collection<Long> adGroupIds,
                                                              @Nullable AdsCountCriteria adsCountCriteria) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.PID, DSL.count())
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds))
                .and(getCondition(adsCountCriteria))
                .groupBy(BANNERS.PID)
                .fetchMap(BANNERS.PID, Record2::value2);
    }

    /**
     * Выбирать из списка кампаний такие, в которых произошли изменения в баннерах, начиная с
     * указанного момента времени.
     *
     * @param shard        шард
     * @param campaignIds  список id кампаний извлекаемых баннеров
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения в баннерах
     */
    public List<Long> getChangedCampaignIds(
            int shard,
            Collection<Long> campaignIds,
            LocalDateTime fromDateTime
    ) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.CID)
                .from(BANNERS)
                .leftJoin(BANNER_IMAGES).using(Tables.BANNERS.BID)
                .where(BANNERS.CID.in(campaignIds))
                .and(BANNERS.LAST_CHANGE.ge(fromDateTime)
                        .or(BANNER_IMAGES.DATE_ADDED.ge(fromDateTime))
                )
                .fetch(BANNERS.CID);
    }

    public Map<Long, BannerAndPhrasesTypes> getBidToTypeMap(int shard, Collection<Long> bids) {
        return dslContextProvider
                .ppc(shard)
                .select(BANNERS.BID, BANNERS.BANNER_TYPE, PHRASES.ADGROUP_TYPE)
                .from(BANNERS)
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .where(BANNERS.BID.in(bids))
                .fetchMap(el -> el.get(BANNERS.BID),
                        el -> new BannerAndPhrasesTypes(el.get(PHRASES.ADGROUP_TYPE),
                                el.get(BANNERS.BANNER_TYPE)));
    }

    public static class BannerAndPhrasesTypes {
        private final PhrasesAdgroupType phrasesAdgroupType;
        private final BannersBannerType bannersBannerType;

        public PhrasesAdgroupType getPhrasesAdgroupType() {
            return phrasesAdgroupType;
        }

        public BannersBannerType getBannersBannerType() {
            return bannersBannerType;
        }

        public BannerAndPhrasesTypes(PhrasesAdgroupType phrasesAdgroupType, BannersBannerType bannersBannerType) {
            this.phrasesAdgroupType = phrasesAdgroupType;
            this.bannersBannerType = bannersBannerType;
        }
    }

    /**
     * Получение счетчиков метрики по списку id баннеров
     * в полученной Map'е для bid без счётчиков значениями будут пустые списки
     */
    public Map<Long, List<Long>> getMetrikaCountersByBids(int shard, Collection<Long> bids) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS)
                .from(BANNERS)
                .leftJoin(CAMP_METRIKA_COUNTERS)
                .on(BANNERS.CID.eq(CAMP_METRIKA_COUNTERS.CID))
                .where(BANNERS.BID.in(bids))
                .fetchMap(BANNERS.BID, r -> Optional.ofNullable(r.get(CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS))
                        .map(CampaignMappings::metrikaCountersFromDb)
                        .orElse(new ArrayList<>()));
    }

    public List<Long> filterBannersWithPublisherId(int shard, Collection<Long> bids) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_PUBLISHER.BID)
                .from(BANNER_PUBLISHER)
                .where(BANNER_PUBLISHER.ZEN_PUBLISHER_ID.isNotNull())
                .and(BANNER_PUBLISHER.BID.in(bids))
                .fetch(BANNER_PUBLISHER.BID);
    }

    private static Condition getCondition(@Nullable AdsCountCriteria countCriteria) {
        Condition condition = DSL.noCondition();
        if (countCriteria == null) {
            return condition;
        }

        if (!countCriteria.getExcludedTypes().isEmpty()) {
            condition = condition.and(BANNERS.BANNER_TYPE.notIn(countCriteria.getExcludedTypes()));
        }

        return condition;
    }
}
